<?php

namespace Drupal\commerce_opp;

use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Represents a supported brand.
 */
final class Brand {

  /**
   * The "Bank Account Brands" type.
   */
  const TYPE_BANK = 'bank';

  /**
   * The "Card Account Brands" type.
   */
  const TYPE_CARD = 'card';

  /**
   * The "Virtual Account Brands" type.
   */
  const TYPE_VIRTUAL = 'virtual';

  /**
   * The brand ID used by Open Payment Platform.
   *
   * @var string
   */
  protected string $id;

  /**
   * The brand ID used by commerce_payment, if available (Credit cards only).
   *
   * @var string
   */
  protected string $commerceId;

  /**
   * The type of the brand: one of "bank", "card", "virtual".
   *
   * @var string
   */
  protected string $type;

  /**
   * The brand label.
   *
   * @var string
   */
  protected string $label;

  /**
   * Whether the workflow is sync/async.
   *
   * @var bool
   */
  protected bool $sync = FALSE;

  /**
   * Whether there is a dedicated gateway plugin for this brand.
   *
   * @var bool
   */
  protected bool $dedicatedPlugin = FALSE;

  /**
   * Whether pre-authorization is supported.
   *
   * @var bool
   */
  protected bool $pa;

  /**
   * Whether direct debit (without pre-authorization) is supported.
   *
   * @var bool
   */
  protected bool $db;

  /**
   * Constructs a new Brand instance.
   *
   * @param array $definition
   *   The definition.
   */
  public function __construct(array $definition) {
    foreach (['id', 'label', 'type'] as $required_property) {
      if (empty($definition[$required_property])) {
        throw new \InvalidArgumentException(new TranslatableMarkup('Missing required property @property.', [
          '@property' => $required_property,
        ]));
      }
    }

    $this->id = $definition['id'];
    $this->label = $definition['label'];
    $this->type = $definition['type'];
    $this->pa = (bool) $definition['pa'];
    $this->db = (bool) $definition['db'];

    if (isset($definition['commerce_id'])) {
      $this->commerceId = $definition['commerce_id'];
    }

    if (isset($definition['workflow'])) {
      $this->sync = $definition['workflow'] === 'sync';
    }

    if (isset($definition['dedicated_plugin'])) {
      $this->dedicatedPlugin = $definition['dedicated_plugin'];
    }
  }

  /**
   * Gets the brand ID used by Open Payment Platform.
   *
   * @return string
   *   The brand ID used by Open Payment Platform.
   */
  public function getId(): string {
    return $this->id;
  }

  /**
   * Gets the brand label.
   *
   * @return string
   *   The brand label.
   */
  public function getLabel(): string {
    return $this->label;
  }

  /**
   * Get the brand ID used by commerce_payment, if available (credit cards).
   *
   * @return string
   *   The brand ID used by commerce_payment, if available (credit cards only).
   */
  public function getCommerceId(): string {
    return $this->commerceId;
  }

  /**
   * Gets the brand type.
   *
   * @return string
   *   The brand type.
   */
  public function getType(): string {
    return $this->type;
  }

  /**
   * Returns whether the workflow is sync or async..
   *
   * @return bool
   *   TRUE, if the workflow is sync, FALSE otherwise.
   */
  public function isSync(): bool {
    return $this->sync;
  }

  /**
   * Returns whether the brand has a dedicated gateway plugin.
   *
   * @return bool
   *   TRUE, if the brand has a dedicated gateway plugin, FALSE otherwise.
   */
  public function isDedicatedPlugin(): bool {
    return $this->dedicatedPlugin;
  }

  /**
   * Returns whether pre-authorization workflow is supported.
   *
   * @return bool
   *   TRUE, if pre-authorization workflow is supported. FALSE otherwise.
   */
  public function supportsAuthorization(): bool {
    return $this->pa;
  }

  /**
   * Returns whether direct debit (without pre-authorization) is supported.
   *
   * @return bool
   *   TRUE, if direct debit (without pre-authorization) workflow is supported.
   *   FALSE otherwise.
   */
  public function supportsDirectDebit(): bool {
    return $this->db;
  }

  /**
   * Gets the array representation of the brand.
   *
   * @return array
   *   The array representation of the brand.
   */
  public function toArray() {
    $values = [
      'id' => $this->getId(),
      'label' => $this->getLabel(),
      'type' => $this->getType(),
      'sync' => $this->isSync(),
      'pa' => $this->pa,
      'db' => $this->db,
    ];
    if (!empty($this->commerceId)) {
      $values['commerce_id'] = $this->commerceId;
    }
    if (isset($this->dedicated_plugin)) {
      $values['dedicated_plugin'] = $this->dedicatedPlugin;
    }
    return $values;
  }

}
