<?php

namespace Drupal\commerce_opp;

use Drupal\Component\Serialization\Json;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Default brand repository implementation.
 */
class BrandRepository implements BrandRepositoryInterface {

  /**
   * The instantiated brands.
   *
   * @var \Drupal\commerce_opp\Brand[]
   */
  protected array $brands = [];

  /**
   * Constructs a new BrandRepository object.
   */
  public function __construct() {
    // JSON source: https://docs.prtso.com/brands/json/index.php.
    $json_path = \Drupal::service('extension.list.module')->getPath('commerce_opp') . DIRECTORY_SEPARATOR . 'data' . DIRECTORY_SEPARATOR . 'payment-methods.json';
    $payment_methods = Json::decode(file_get_contents($json_path));

    $label_mapping = [
      'AMEX' => 'American Express (AMEX)',
      'DISCOVER' => 'Discover Card',
      'EPS' => 'eps-Überweisung',
      'MAESTRO' => 'Maestro',
      'MASTER' => 'MasterCard',
      'MASTERDEBIT' => 'MasterCard Debit',
      'PAYPAL' => 'PayPal',
      'RATENKAUF' => 'ratenkauf by easyCredit',
      'SOFORTUEBERWEISUNG' => 'SOFORT Überweisung',
    ];

    $dedicated_plugins = [
      'MBWAY',
      'SOFORTUEBERWEISUNG',
      'SIBS_MULTIBANCO',
    ];

    $commerce_id_mapping = [
      'AMEX' => 'amex',
      'CHINAUNIONPAY' => 'unionpay',
      'DINERS' => 'dinersclub',
      'DISCOVER' => 'discover',
      'JCB' => 'jcb',
      'MAESTRO' => 'maestro',
      'MASTER' => 'mastercard',
      'VISA' => 'visa',
    ];

    foreach ($payment_methods as $payment_method) {
      $id = $payment_method['brand'];
      $payment_method['id'] = $id;
      $payment_method['label'] = $label_mapping[$id] ?? $id;
      $payment_method['dedicated_plugin'] = in_array($id, $dedicated_plugins);
      $payment_method['commerce_id'] = $commerce_id_mapping[$id] ?? '';
      $this->brands[$id] = new Brand($payment_method);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getBrand($id) {
    if (!isset($this->brands[$id])) {
      throw new \InvalidArgumentException(new TranslatableMarkup('Invalid brand "@brand_id"', [
        '@brand_id' => $id,
      ]));
    }
    return $this->brands[$id];
  }

  /**
   * {@inheritdoc}
   */
  public function getBrands() {
    return $this->brands;
  }

  /**
   * {@inheritdoc}
   */
  public function getBrandLabels() {
    $brands = $this->getBrands();
    $brand_labels = array_map(function ($brand) {
      /** @var \Drupal\commerce_opp\Brand $brand */
      return $brand->getLabel();
    }, $brands);
    return $brand_labels;
  }

  /**
   * {@inheritdoc}
   */
  public function getCardAccountBrands($exclude_dedicated_plugins = TRUE) {
    $brands = $this->getBrands();
    $filtered_brands = array_filter($brands, function ($brand) {
      /** @var \Drupal\commerce_opp\Brand $brand */
      return $brand->getType() == Brand::TYPE_CARD;
    });
    if ($exclude_dedicated_plugins) {
      $filtered_brands = array_filter($filtered_brands, function ($brand) {
        /** @var \Drupal\commerce_opp\Brand $brand */
        return !$brand->isDedicatedPlugin();
      });
    }
    return $filtered_brands;
  }

  /**
   * {@inheritdoc}
   */
  public function getCardAccountBrandLabels($exclude_dedicated_plugins = TRUE) {
    $brands = $this->getCardAccountBrands($exclude_dedicated_plugins);
    $brand_labels = array_map(function ($brand) {
      /** @var \Drupal\commerce_opp\Brand $brand */
      return $brand->getLabel();
    }, $brands);
    return $brand_labels;
  }

  /**
   * {@inheritdoc}
   */
  public function getBankAccountBrands($exclude_dedicated_plugins = TRUE) {
    $brands = $this->getBrands();
    $filtered_brands = array_filter($brands, function ($brand) {
      /** @var \Drupal\commerce_opp\Brand $brand */
      return $brand->getType() == Brand::TYPE_BANK;
    });
    if ($exclude_dedicated_plugins) {
      $filtered_brands = array_filter($filtered_brands, function ($brand) {
        /** @var \Drupal\commerce_opp\Brand $brand */
        return !$brand->isDedicatedPlugin();
      });
    }
    return $filtered_brands;
  }

  /**
   * {@inheritdoc}
   */
  public function getBankAccountBrandLabels($exclude_dedicated_plugins = TRUE) {
    $brands = $this->getBankAccountBrands($exclude_dedicated_plugins);
    $brand_labels = array_map(function ($brand) {
      /** @var \Drupal\commerce_opp\Brand $brand */
      return $brand->getLabel();
    }, $brands);
    return $brand_labels;
  }

  /**
   * {@inheritdoc}
   */
  public function getVirtualAccountBrands($exclude_dedicated_plugins = TRUE) {
    $brands = $this->getBrands();
    $filtered_brands = array_filter($brands, function ($brand) {
      /** @var \Drupal\commerce_opp\Brand $brand */
      return $brand->getType() == Brand::TYPE_VIRTUAL;
    });
    if ($exclude_dedicated_plugins) {
      $filtered_brands = array_filter($filtered_brands, function ($brand) {
        /** @var \Drupal\commerce_opp\Brand $brand */
        return !$brand->isDedicatedPlugin();
      });
    }
    return $filtered_brands;
  }

  /**
   * {@inheritdoc}
   */
  public function getVirtualAccountBrandLabels($exclude_dedicated_plugins = TRUE) {
    $brands = $this->getVirtualAccountBrands($exclude_dedicated_plugins);
    $brand_labels = array_map(function ($brand) {
      /** @var \Drupal\commerce_opp\Brand $brand */
      return $brand->getLabel();
    }, $brands);
    return $brand_labels;
  }

}
