<?php

namespace Drupal\commerce_opp\PluginForm;

use CommerceGuys\Intl\Formatter\CurrencyFormatterInterface;
use Drupal\commerce_opp\Plugin\Commerce\PaymentGateway\CopyAndPaySofortueberweisungInterface;
use Drupal\commerce_opp\Plugin\Commerce\PaymentGateway\CopyAndPaySibsMultibanco;
use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the COPYandPAY plugin form.
 */
class CopyAndPayForm extends PaymentOffsiteForm implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The currency formatter.
   *
   * @var \CommerceGuys\Intl\Formatter\CurrencyFormatterInterface
   */
  protected CurrencyFormatterInterface $currencyFormatter;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Constructs a new CopyAndPayForm object.
   *
   * @param \CommerceGuys\Intl\Formatter\CurrencyFormatterInterface $currency_formatter
   *   The currency formatter.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   */
  public function __construct(CurrencyFormatterInterface $currency_formatter, LanguageManagerInterface $language_manager) {
    $this->currencyFormatter = $currency_formatter;
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('commerce_price.currency_formatter'),
      $container->get('language_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->entity;
    $order = $payment->getOrder();
    $customer = $order->getCustomer();

    /** @var \Drupal\commerce_opp\Plugin\Commerce\PaymentGateway\CopyAndPayInterface $opp_gateway */
    $opp_gateway = $payment->getPaymentGateway()->getPlugin();
    $brand_ids = $opp_gateway->getBrandIds();

    $payment_amount = $opp_gateway->getPayableAmount($order);
    $payment->setAmount($payment_amount);
    // Save to get an ID.
    $payment->save();

    $request_params = $opp_gateway->prepareCheckoutRequestParameters($payment, $payment_amount);

    $payment_method = $payment->getPaymentMethod();
    if ($payment_method && $payment_method->getRemoteId()) {
      $request_params['standingInstruction.mode'] = 'REPEATED';
      $request_params['standingInstruction.source'] = 'MIT';
      $request_params['standingInstruction.type'] = 'UNSCHEDULED';
      $request_params['registrations[0].id'] = $payment_method->getRemoteId();
    }
    elseif ($customer && $customer->isAuthenticated() && $opp_gateway->allowRegistrations()) {
      $request_params['createRegistration'] = TRUE;
      $request_params['standingInstruction.mode'] = 'INITIAL';
      $request_params['standingInstruction.source'] = 'CIT';
      $request_params['standingInstruction.type'] = 'UNSCHEDULED';
    }

    $checkout_id = $opp_gateway->prepareCheckout($request_params, $payment);

    $is_sibs_multibanco = $opp_gateway instanceof CopyAndPaySibsMultibanco;
    if (!$is_sibs_multibanco) {
      $payment->setExpiresTime($opp_gateway->calculateCheckoutIdExpireTime());
    }
    $payment->save();

    $script_url = sprintf("%s/v1/paymentWidgets.js?checkoutId=%s", $opp_gateway->getActiveHostUrl(), $checkout_id);
    $js_settings = [
      'opp_script_url' => $script_url,
      'langcode' => $this->languageManager->getCurrentLanguage()->getId(),
    ];
    if ($opp_gateway instanceof CopyAndPaySofortueberweisungInterface) {
      $sofort_countries = $opp_gateway->getSofortCountries();
      if ($opp_gateway->isSofortRestrictedToBillingAddress() && !empty($billing_address)) {
        $restrict_countries = [strtoupper($billing_address->getCountryCode())];
        $restrict_countries = array_combine($restrict_countries, $restrict_countries);
        $sofort_countries = array_intersect_key($sofort_countries, $restrict_countries);
      }
      $js_settings['sofort_countries'] = (object) $sofort_countries;
    }
    $form['#attached']['drupalSettings']['commerce_opp'] = $js_settings;
    $form['#attached']['library'][] = 'commerce_opp/init';

    $amount_formatted = '';
    if ($opp_gateway->isAmountVisible()) {
      $amount_formatted = $this->t('Amount to be paid: @amount', [
        '@amount' => $this->currencyFormatter->format($payment_amount->getNumber(), $payment_amount->getCurrencyCode()),
      ]);
    }

    $form['cards'] = [
      '#type' => 'hidden',
      '#value' => implode(' ', $brand_ids),
      // Plugin forms are embedded using #process, so it's too late to attach
      // another #process to $form itself, it must be on a sub-element.
      '#process' => [
        [get_class($this), 'processCopyAndPayForm'],
      ],
      '#action' => $form['#return_url'],
      '#cancel_url' => $form['#cancel_url'],
      '#amount' => $amount_formatted,
    ];

    // No need to call buildRedirectForm(), as we embed an iframe.
    return $form;
  }

  /**
   * Prepares the complete form in order to work with COPYandPAY.
   *
   * Sets the form #action, adds a class for the JS to target.
   * Workaround for buildConfigurationForm() not receiving $complete_form.
   *
   * @param array $element
   *   The form element whose value is being processed.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param array $complete_form
   *   The complete form structure.
   *
   * @return array
   *   The processed form element.
   */
  public static function processCopyAndPayForm(array $element, FormStateInterface $form_state, array &$complete_form) {
    $complete_form['#action'] = $element['#action'];
    $complete_form['#attributes']['class'][] = 'paymentWidgets';
    $complete_form['#attributes']['data-brands'] = $element['#value'];

    if (!empty($element['#amount'])) {
      $complete_form['#prefix'] = $element['#amount'];
    }

    // As the COPYandPAY fully replaces the HTML form, we need to place the
    // cancel link outside the form as suffix.
    $complete_form['#suffix'] = Link::fromTextAndUrl(t('Cancel'), Url::fromUri($element['#cancel_url']))->toString();

    return $element;
  }

}
