<?php

namespace Drupal\commerce_opp\Transaction\Status;

use Drupal\commerce_opp\Brand;
use Drupal\commerce_price\Price;

/**
 * Abstract transaction status base class.
 */
abstract class AbstractStatus implements TransactionStatusInterface {

  /**
   * The transaction ID.
   *
   * @var string
   */
  protected string $id;

  /**
   * The result code.
   *
   * @var string
   */
  protected string $code;

  /**
   * The status description.
   *
   * @var string
   */
  protected string $description;

  /**
   * The payment brand.
   *
   * @var \Drupal\commerce_opp\Brand|null
   */
  protected ?Brand $brand = NULL;

  /**
   * The payment amount.
   *
   * @var \Drupal\commerce_price\Price|null
   */
  protected ?Price $price = NULL;

  /**
   * The payment type.
   *
   * @var string|null
   */
  protected ?string $paymentType = NULL;

  /**
   * The registration ID.
   *
   * @var string|null
   */
  protected ?string $registrationId = NULL;

  /**
   * The (credit) card info.
   *
   * @var array
   */
  protected array $card = [];

  /**
   * The virtual account info.
   *
   * @var array
   */
  protected array $virtualAccount = [];

  /**
   * The bank account info.
   *
   * @var array
   */
  protected array $bankAccount = [];

  /**
   * Constructs a new AbstractStatus object.
   *
   * @param string $id
   *   The transaction ID.
   * @param string $code
   *   The result code.
   * @param string $description
   *   The status description.
   * @param \Drupal\commerce_opp\Brand|null $brand
   *   The payment brand. Defaults to NULL.
   */
  public function __construct(string $id, string $code, string $description, ?Brand $brand = NULL) {
    $this->id = $id;
    $this->code = $code;
    $this->description = $description;
    $this->brand = $brand;
  }

  /**
   * {@inheritdoc}
   */
  public function getId(): string {
    return $this->id;
  }

  /**
   * {@inheritdoc}
   */
  public function getCode(): string {
    return $this->code;
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string {
    return $this->description;
  }

  /**
   * {@inheritdoc}
   */
  public function getBrand(): ?Brand {
    return $this->brand;
  }

  /**
   * {@inheritdoc}
   */
  public function isAsyncPayment(): bool {
    return $this->brand && !$this->brand->isSync();
  }

  /**
   * {@inheritdoc}
   */
  public function getPrice(): ?Price {
    return $this->price;
  }

  /**
   * {@inheritdoc}
   */
  public function setPrice(Price $price): void {
    $this->price = $price;
  }

  /**
   * {@inheritdoc}
   */
  public function getPaymentType(): ?string {
    return $this->paymentType;
  }

  /**
   * {@inheritdoc}
   */
  public function setPaymentType(string $paymentType): void {
    $this->paymentType = $paymentType;
  }

  /**
   * {@inheritdoc}
   */
  public function getRegistrationId(): ?string {
    return $this->registrationId;
  }

  /**
   * {@inheritdoc}
   */
  public function setRegistrationId(string $registrationId): void {
    $this->registrationId = $registrationId;
  }

  /**
   * {@inheritdoc}
   */
  public function getCardInfo(): array {
    return $this->card;
  }

  /**
   * {@inheritdoc}
   */
  public function setCardInfo(array $card_info): void {
    $this->card = $card_info;
  }

  /**
   * {@inheritdoc}
   */
  public function getVirtualAccountInfo(): array {
    return $this->virtualAccount;
  }

  /**
   * {@inheritdoc}
   */
  public function setVirtualAccountInfo(array $va_info): void {
    $this->virtualAccount = $va_info;
  }

  /**
   * {@inheritdoc}
   */
  public function getBankAccountInfo(): array {
    return $this->bankAccount;
  }

  /**
   * {@inheritdoc}
   */
  public function setBankAccountInfo(array $ba_info): void {
    $this->bankAccount = $ba_info;
  }

  /**
   * {@inheritdoc}
   */
  public function toArray(): array {
    $values = [
      'id' => $this->id,
      'result' => [
        'code' => $this->code,
        'description' => $this->description,
      ],
      'brand' => $this->brand?->toArray(),
      'paymentType' => $this->paymentType,
      'registrationId' => $this->registrationId,
      'card' => $this->card,
      'virtualAccount' => $this->virtualAccount,
      'bankAccount' => $this->bankAccount,
    ];

    if ($price = $this->getPrice()) {
      $values['amount'] = $price->getNumber();
      $values['currency'] = $price->getCurrencyCode();
    }

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  abstract public function getType(): string;

}
