<?php

namespace Drupal\commerce_opp\Transaction\Status;

use Drupal\commerce_opp\Brand;
use Drupal\commerce_price\Price;

/**
 * Defines the transaction status interface.
 */
interface TransactionStatusInterface {

  /**
   * Gets the transaction ID.
   *
   * @return string
   *   The transaction ID.
   */
  public function getId(): string;

  /**
   * Gets the result code.
   *
   * @return string
   *   The result code.
   */
  public function getCode(): string;

  /**
   * Gets the status description.
   *
   * @return string
   *   The status description.
   */
  public function getDescription(): string;

  /**
   * Gets the type of this status.
   *
   * The result is one of the TYPE_* constants of the Constants class. As the
   * class of the given instance already implies the status type, this can be
   * seen as redundant information for convenience.
   *
   * @return string
   *   The status type (whether if the transaction was successful, failed, etc).
   */
  public function getType(): string;

  /**
   * Gets the payment brand.
   *
   * @return \Drupal\commerce_opp\Brand|null
   *   The payment brand.
   */
  public function getBrand(): ?Brand;

  /**
   * Gets whether the payment is async.
   *
   * It always depends on the brand, if the payment type is sync or async.
   *
   * @return bool
   *   TRUE, if the associated payment brand is an async one, FALSE otherwise.
   */
  public function isAsyncPayment(): bool;

  /**
   * Gets the array representation of the transaction status.
   *
   * @return array
   *   The array representation of the transaction status.
   */
  public function toArray(): array;

  /**
   * Get the price (payment amount).
   *
   * @return \Drupal\commerce_price\Price|null
   *   The price (payment amount).
   */
  public function getPrice(): ?Price;

  /**
   * Set the price (payment amount).
   *
   * @param \Drupal\commerce_price\Price $price
   *   The price (payment amount).
   */
  public function setPrice(Price $price): void;

  /**
   * Get the payment type.
   *
   * @return string|null
   *   The payment type.
   */
  public function getPaymentType(): ?string;

  /**
   * Set the payment type.
   *
   * @param string $paymentType
   *   The payment type.
   */
  public function setPaymentType(string $paymentType): void;

  /**
   * Get the registration ID.
   *
   * @return string|null
   *   The registration ID.
   */
  public function getRegistrationId(): ?string;

  /**
   * Set the registration ID.
   *
   * @param string $registrationId
   *   The registration ID.
   */
  public function setRegistrationId(string $registrationId): void;

  /**
   * Get (credit) card info, if available.
   *
   * @return array
   *   The (credit) card info, if available.
   */
  public function getCardInfo(): array;

  /**
   * Set (credit) card info.
   *
   * @param array $card_info
   *   The (credit) card info.
   */
  public function setCardInfo(array $card_info): void;

  /**
   * Get virtual account info, if available.
   *
   * @return array
   *   The virtual account info, if available.
   */
  public function getVirtualAccountInfo(): array;

  /**
   * Set virtual account info.
   *
   * @param array $va_info
   *   The virtual account info.
   */
  public function setVirtualAccountInfo(array $va_info): void;

  /**
   * Get bank account info, if available.
   *
   * @return array
   *   The bank account info, if available.
   */
  public function getBankAccountInfo(): array;

  /**
   * Set bank account info.
   *
   * @param array $ba_info
   *   The bank account info.
   */
  public function setBankAccountInfo(array $ba_info): void;

}
