<?php

namespace Drupal\commerce_order_item_sku\EventSubscriber;

use Drupal\commerce_cart\Event\CartEvents;
use Drupal\commerce_cart\Event\CartOrderItemAddEvent;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\state_machine\Event\WorkflowTransitionEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for Commerce Order events.
 */
class OrderItemSkuSubscriber implements EventSubscriberInterface {
  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new OrderItemSkuSubscriber object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      'commerce_order.place.post_transition' => 'onOrderPlace',
      CartEvents::CART_ORDER_ITEM_ADD => 'onOrderItemAdd',
    ];
  }

  /**
   * Reacts to the order place event.
   *
   * @param \Drupal\state_machine\Event\WorkflowTransitionEvent $event
   *   The transition event.
   */
  public function onOrderPlace(WorkflowTransitionEvent $event) {
    $config = $this->configFactory->get('commerce_order_item_sku.settings');
    if ($config->get('setting_event') !== 'order_placed') {
      return;
    }

    $order = $event->getEntity();
    foreach ($order->getItems() as $order_item) {
      $purchased_entity = $order_item->getPurchasedEntity();
      if (!$purchased_entity->hasField('sku')
        || $purchased_entity->get('sku')->isEmpty()) {
        continue;
      }

      $sku = $purchased_entity->get('sku')->value;
      $order_item->set('sku', $sku);
      $order_item->save();
    }
  }

  /**
   * Reacts to the order item add event.
   *
   * @param \Drupal\commerce_cart\Event\CartOrderItemAddEvent $event
   *   The order item add event.
   */
  public function onOrderItemAdd(CartOrderItemAddEvent $event) {
    $config = $this->configFactory->get('commerce_order_item_sku.settings');
    if ($config->get('setting_event') !== 'order_item_add') {
      return;
    }

    $order_item = $event->getOrderItem();
    $purchased_entity = $order_item->getPurchasedEntity();
    if (!$purchased_entity->hasField('sku')
      || $purchased_entity->get('sku')->isEmpty()) {
      return;
    }
    $sku = $purchased_entity->get('sku')->value;
    $order_item->set('sku', $sku);
    $order_item->save();
  }

}
