<?php

namespace Drupal\commerce_order_item_sku\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure settings for the Commerce Order Item SKU module.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['commerce_order_item_sku.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'commerce_order_item_sku_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config('commerce_order_item_sku.settings');

    $form['setting_event_title'] = [
      '#type' => 'item',
      '#markup' => $this->t('SKU setting event'),
    ];

    $options = [
      'order_item_add' => [
        'title' => $this->t('Order item cart add'),
        'description' => $this->t(
          'Automatically sets the SKU on item cart add.'
        ),
      ],
      'order_item_presave' => [
        'title' => $this->t('Order item saving'),
        'description' => $this->t(
          'Automatically sets the SKU on new item saving.'
        ),
      ],
      'order_placed' => [
        'title' => $this->t('Order placement'),
        'description' => $this->t(
          'Automatically sets the SKU on order placement.'
        ),
      ],
      'purchased_entity_delete' => [
        'title' => $this->t('Purchased entity deletion'),
        'description' => $this->t(
          'Automatically sets the SKU on purchased entity deletion form submit.'
        ),
      ],
      'none' => [
        'title' => $this->t('None'),
        'description' => $this->t(
          'Do not automatically sets the SKU.<br> You can implement your own logic or just proceed manualy.'
        ),
      ],
    ];

    $default_value = $config->get('setting_event') ?? 'order_item_presave';

    foreach ($options as $value => $data) {
      $form['setting_event'][$value] = [
        '#type' => 'radio',
        '#title' => $data['title'],
        '#description' => $data['description'],
        '#name' => 'setting_event',
        '#default_value' => $default_value,
        '#return_value' => $value,
        '#parents' => ['setting_event'],
      ];
    }

    $form['sync_on_purchased_entity_reference_update'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Automatically update SKU on purchased entity reference update'),
      '#description' => $this->t(
        'Synchronization will be skipped if the order item SKU does not match the original purchased item SKU anymore.'
      ),
      '#default_value' => $config->get('sync_on_purchased_entity_reference_update') ?? FALSE,
    ];

    $form['sync_on_purchasable_entity_sku_update'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Automatically update SKU on purchased entity SKU update'),
      '#description' => $this->t(
        'Synchronization will happen on purchased entity update form submit.<br>'
        . 'It will be skipped if the order item SKU does not match the original purchased item SKU anymore.'
      ),
      '#default_value' => $config->get('sync') ?? FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->configFactory->getEditable('commerce_order_item_sku.settings')
      ->set('setting_event', $form_state->getValue('setting_event'))
      ->set('sync_on_purchasable_entity_sku_update', $form_state->getValue('sync_on_purchasable_entity_sku_update'))
      ->set('sync_on_purchased_entity_reference_update', $form_state->getValue('sync_on_purchased_entity_reference_update'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
