<?php

declare(strict_types=1);

namespace Drupal\Tests\commerce_order_item_sku\FunctionalJavascript;

use Drupal\commerce_order\Entity\OrderItem;
use Drupal\commerce_order\Entity\OrderItemType;
use Drupal\Tests\commerce\FunctionalJavascript\CommerceWebDriverTestBase;

/**
 * Tests order item SKU batch update on purchasable entity SKU update.
 *
 * @group commerce_order_item_sku
 */
class PurchasableEntityDeleteTest extends CommerceWebDriverTestBase {

  /**
   * A test order.
   *
   * @var \Drupal\commerce_order\Entity\OrderInterface
   */
  protected $order;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'commerce_product',
    'commerce_order',
    'commerce_order_item_sku',
    'commerce_order_item_sku_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected function getAdministratorPermissions() {
    return [
      'administer commerce_product',
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $order_item_type = OrderItemType::load('default');
    $order_item_type->setTraits(['commerce_order_item_sku']);
    $order_item_type->save();

    $trait_manager = $this->container->get('plugin.manager.commerce_entity_trait');
    $trait = $trait_manager->createInstance('commerce_order_item_sku');
    $trait_manager->installTrait($trait, 'commerce_order_item', 'default');

    $configFactory = $this->container->get('config.factory');
    $configFactory->getEditable('commerce_order_item_sku.settings')
      ->set('setting_event', 'purchased_entity_delete')
      ->set('sync_on_purchasable_entity_sku_update', FALSE)
      ->set('sync_on_purchased_entity_reference_update', FALSE)
      ->save();

    $sku = 'ORIGINAL_SKU';
    $variation = $this->createEntity('commerce_product_variation', [
      'type' => 'default',
      'sku' => $sku,
      'price' => [
        'number' => 999,
        'currency_code' => 'USD',
      ],
    ]);

    $this->createEntity('commerce_product', [
      'type' => 'default',
      'title' => 'Product 1',
      'stores' => [$this->store],
      'variations' => [$variation],
    ]);

    $order_items = [];
    for ($i = 0; $i < 3; $i++) {
      $order_items[] = $this->createEntity('commerce_order_item', [
        'type' => 'default',
        'purchased_entity' => $variation->id(),
      ]);
    }
    $order_items[0]->set('sku', 'UPDATED_SKU');
    $order_items[0]->save();

    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $this->order = $this->createEntity('commerce_order', [
      'type' => 'default',
      'mail' => $this->loggedInUser->getEmail(),
      'uid' => $this->loggedInUser->id(),
      'order_items' => $order_items,
      'store_id' => $this->store,
    ]);
  }

  /**
   * Tests purchasable entity deletion.
   */
  public function testEntityDelete(): void {
    $this->drupalGet('product/1/variations/1/delete');
    $this->submitForm([], 'Delete');
    $this->assertSession()->pageTextContains('Updating order items');
    $this->assertSession()->waitForText('The Product 1 variation has been deleted.');
    $this->assertSession()->pageTextContains('2 items successfully processed.');
    $order_items = OrderItem::loadMultiple();
    $first_item = array_shift($order_items);
    $this->assertEquals('UPDATED_SKU', $first_item->get('sku')->value, 'The order item SKU should be updated.');
    foreach ($order_items as $order_item) {
      $this->assertEquals('ORIGINAL_SKU', $order_item->get('sku')->value, 'The order item SKU should be updated.');
    }
  }

}
