<?php

declare(strict_types=1);

namespace Drupal\commerce_pay_publish\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Controller for handling add to cart functionality.
 */
class CommercePayPublishAddToCartController extends ControllerBase {

  /**
   * Adds a product to cart and redirects to checkout.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to get the associated product from.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect to checkout or product page.
   */
  public function addToCart(NodeInterface $node) {
    // Get the product associated with this node.
    $product = $this->getProductFromNode($node);

    if (!$product) {
      $this->messenger()->addError($this->t('No product found for this content.'));
      return new RedirectResponse(Url::fromRoute('entity.node.canonical', ['node' => $node->id()])->toString());
    }

    // Get the first variation of the product.
    $variations = $product->getVariations();
    if (empty($variations)) {
      $this->messenger()->addError($this->t('No product variations found.'));
      return new RedirectResponse(Url::fromRoute('entity.node.canonical', ['node' => $node->id()])->toString());
    }

    $variation = reset($variations);

    // Clear the cart first to ensure only one product.
    $this->clearCart();

    // Add the product to cart using the cart manager.
    $cart_provider = \Drupal::service('commerce_cart.cart_provider');
    $cart_manager = \Drupal::service('commerce_cart.cart_manager');

    $cart = $cart_provider->getCart('default');
    if (!$cart) {
      $cart = $cart_provider->createCart('default');
    }

    // Add the product variation to cart.
    $order_item = $cart_manager->addEntity($cart, $variation, 1);

    // Add the node ID to the order item for tracking.
    if ($order_item && $order_item->hasField('field_source_nid')) {
      $order_item->set('field_source_nid', $node->id());
      $order_item->save();
    }

    // Redirect to checkout.
    $checkout_url = Url::fromRoute('commerce_checkout.form', ['commerce_order' => $cart->id()])->toString();
    return new RedirectResponse($checkout_url);
  }

  /**
   * Gets the product associated with a node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node.
   *
   * @return \Drupal\commerce_product\Entity\ProductInterface|null
   *   The product or null if not found.
   */
  protected function getProductFromNode(NodeInterface $node) {
    // First try to get from the field_pay_publish_product field.
    if ($node->hasField('field_pay_publish_product') && !$node->get('field_pay_publish_product')->isEmpty()) {
      return $node->get('field_pay_publish_product')->entity;
    }

    // Fallback to the custom table if the field doesn't exist.
    $connection = \Drupal::database();
    $product_id = $connection
      ->select('commerce_pay', 'cp')
      ->fields('cp', ['pid'])
      ->condition('nid', $node->id())
      ->execute()
      ->fetchField();

    if ($product_id) {
      return \Drupal::entityTypeManager()->getStorage('commerce_product')->load($product_id);
    }

    return NULL;
  }

  /**
   * Clears the current cart.
   */
  protected function clearCart() {
    $cart_provider = \Drupal::service('commerce_cart.cart_provider');
    $carts = $cart_provider->getCarts();

    if (!empty($carts)) {
      foreach ($carts as $cart) {
        foreach ($cart->getItems() as $order_item) {
          $cart->removeItem($order_item);
        }
        $cart->save();
      }
    }
  }

}
