<?php

declare(strict_types=1);

namespace Drupal\commerce_pay_publish\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\NodeType;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure settings for Commerce Pay Publish module.
 */
final class CommercePayPublishSettingsForm extends ConfigFormBase {

  public const CONFIG_NAME = 'commerce_pay_publish.settings';
  private const FIELD_NAME = 'field_pay_publish_product';

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new CommercePayPublishSettingsForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Creates an instance of the service via dependency injection.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   A new instance of the class.
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * Returns the unique ID of the form.
   *
   * @return string
   *   The unique form ID.
   */
  public function getFormId(): string {
    return 'commerce_pay_publish_settings_form';
  }

  /**
   * Returns the editable configuration object names.
   *
   * @return string[]
   *   An array of editable configuration object names.
   */
  protected function getEditableConfigNames(): array {
    return [self::CONFIG_NAME];
  }

  /**
   * Builds the settings form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The modified form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config(self::CONFIG_NAME);

    // Load all node types as options.
    $node_types = NodeType::loadMultiple();
    $options = [];
    foreach ($node_types as $type) {
      $options[$type->id()] = $type->label();
    }

    $form['enabled_node_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Content types requiring payment to publish'),
      '#options' => $options,
      '#default_value' => $config->get('enabled_node_types') ?? [],
      '#description' => $this->t('Select one or more content types that require payment to publish.'),
    ];

    $config_node_products = $config->get('node_type_products') ?? [];


    // Get pay_publish products.
    $product_options = $this->getPayPublishProducts();

    foreach ($options as $node_type_id => $node_type_label) {
      $form['node_type_products_' . $node_type_id] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('@type Products', ['@type' => $node_type_label]),
        '#options' => $product_options,
        '#default_value' => $config_node_products[$node_type_id] ?? [],
        '#states' => [
          'visible' => [
            ':input[name="enabled_node_types[' . $node_type_id . ']"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    // Check if no pay_publish products exist.
    if (empty($product_options)) {
      $form['default_pay_publish_product']['#description'] = $this->t('No pay_publish products found. Please create products of type "Pay Publish" first.');
      $form['default_pay_publish_product']['#disabled'] = TRUE;
    }
    // Add default pay_publish product selectbox.
    $form['default_pay_publish_product'] = [
      '#type' => 'select',
      '#title' => $this->t('Default Pay Publish Product'),
      '#options' => $product_options,
      '#empty_option' => $this->t('- Select a default product -'),
      '#default_value' => $config->get('default_pay_publish_product') ?? '',
      '#description' => $this->t('Select the default pay_publish product which will be consider free/trial product.'),
    ];

    // Notification settings.
    $form['notifications'] = [
      '#type' => 'details',
      '#title' => $this->t('Notification Settings'),
      '#open' => TRUE,
    ];

    $form['notifications']['notifications_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable notifications'),
      '#default_value' => $config->get('notifications_enabled') ?? FALSE,
      '#description' => $this->t('Enable email notifications for publish, expire, and renewal events.'),
    ];

    // Notification fieldsets - only show when notifications are enabled.
    $form['notifications']['notification_settings'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="notifications_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Publish notification fieldset.
    $form['notifications']['notification_settings']['publish_notification'] = [
      '#type' => 'details',
      '#title' => $this->t('Publish Notification'),
      '#open' => TRUE,
    ];

    $form['notifications']['notification_settings']['publish_notification']['publish_email_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $config->get('publish_email_subject') ?? 'Your content has been published',
      '#description' => $this->t('Available tokens: [node:title], [node:url], [node:author:name]'),
      '#required' => FALSE,
    ];

    $form['notifications']['notification_settings']['publish_notification']['publish_email_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#default_value' => $config->get('publish_email_body') ?? "Your content '[node:title]' has been published and will expire on [commerce_pay_publish:expiry_date].\n\nView your content: [node:url]",
      '#description' => $this->t('Available tokens: [node:title], [node:url], [node:author:name]'),
      '#rows' => 5,
      '#required' => FALSE,
    ];

    // Expire notification fieldset.
    $form['notifications']['notification_settings']['expire_notification'] = [
      '#type' => 'details',
      '#title' => $this->t('Expire Notification'),
      '#open' => TRUE,
    ];

    $form['notifications']['notification_settings']['expire_notification']['expire_email_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $config->get('expire_email_subject') ?? 'Your content has expired',
      '#description' => $this->t('Available tokens: [node:title], [node:url], [node:author:name]'),
      '#required' => FALSE,
    ];

    $form['notifications']['notification_settings']['expire_notification']['expire_email_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#default_value' => $config->get('expire_email_body') ?? "Your content '[node:title]' has expired.\n\nYou can relist it by clicking here: [commerce_pay_publish:relist_url]\n\nOr view your content: [node:url]",
      '#description' => $this->t('Available tokens: [node:title], [node:url], [node:author:name]'),
      '#rows' => 5,
      '#required' => FALSE,
    ];

    // Renewal notification fieldset.
    $form['notifications']['notification_settings']['renewal_notification'] = [
      '#type' => 'details',
      '#title' => $this->t('Renewal Notification'),
      '#open' => TRUE,
    ];

    $form['notifications']['notification_settings']['renewal_notification']['renewal_email_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $config->get('renewal_email_subject') ?? 'Your content has been renewed',
      '#description' => $this->t('Available tokens: [node:title], [node:url], [node:author:name]'),
      '#required' => FALSE,
    ];

    $form['notifications']['notification_settings']['renewal_notification']['renewal_email_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#default_value' => $config->get('renewal_email_body') ?? "Your content '[node:title]' has been renewed and will expire on [commerce_pay_publish:expiry_date].\n\nView your content: [node:url]",
      '#description' => $this->t('Available tokens: [node:title], [node:url], [node:author:name]'),
      '#rows' => 5,
      '#required' => FALSE,
    ];

    // Token browser helper.
    $form['notifications']['notification_settings']['publish_token_help'] = [
      '#theme' => 'token_tree_link',
      '#token_types' => ['node', 'commerce_pay_publish_plan', 'commerce_pay_publish_plan_usage'],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Get pay_publish products.
   */
  private function getPayPublishProducts(): array {
    $pay_publish_products = $this->entityTypeManager
      ->getStorage('commerce_product')
      ->loadByProperties(['type' => 'pay_publish']);

    $product_options = [];
    foreach ($pay_publish_products as $product) {
      $product_options[$product->id()] = $product->label();
    }

    return $product_options;
  }

  /**
   * Submits the settings form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config(self::CONFIG_NAME);
    $previous_types = $config->get('enabled_node_types') ?? [];
    $current_types = array_filter($form_state->getValue('enabled_node_types'));
    $node_type_products = [];

    // Save config.
    $config_editable = $this->configFactory()->getEditable(self::CONFIG_NAME);
    $config_editable->set('enabled_node_types', $current_types);

    // Save checkout checkbox value.
    $config_editable->set('default_pay_publish_product', $form_state->getValue('default_pay_publish_product'));

    // Save notification settings.
    $config_editable->set('notifications_enabled', $form_state->getValue('notifications_enabled'));
    $config_editable->set('publish_email_subject', $form_state->getValue('publish_email_subject'));
    $config_editable->set('publish_email_body', $form_state->getValue('publish_email_body'));
    $config_editable->set('expire_email_subject', $form_state->getValue('expire_email_subject'));
    $config_editable->set('expire_email_body', $form_state->getValue('expire_email_body'));
    $config_editable->set('renewal_email_subject', $form_state->getValue('renewal_email_subject'));
    $config_editable->set('renewal_email_body', $form_state->getValue('renewal_email_body'));
    foreach ($form_state->getValue('enabled_node_types') as $bundle_id => $checked) {
      if ($checked && $form_state->hasValue('node_type_products_' . $bundle_id)) {
        $node_type_products[$bundle_id] = array_filter($form_state->getValue('node_type_products_' . $bundle_id));
      }
    }
    $config_editable->set('node_type_products', $node_type_products);

    $config_editable->save();

    // Add field to newly selected types.
    $to_add = array_diff($current_types, $previous_types);
    foreach ($to_add as $bundle) {
      $this->addPayPublishProductField($bundle);
    }

    // Remove field from unselected types.
    $to_remove = array_diff($previous_types, $current_types);
    foreach ($to_remove as $bundle) {
      $this->removePayPublishProductField($bundle);
    }

    // Add or update Source Nid field on commerce_order default bundle.
    $this->addOrUpdateSourceNidFieldOnOrder($current_types);

    parent::submitForm($form, $form_state);
  }

  /**
   * Adds the commerce product reference field to a content type.
   */
  private function addPayPublishProductField(string $bundle): void {
    // Create field storage if it doesn't exist.
    if (!FieldStorageConfig::loadByName('node', self::FIELD_NAME)) {
      FieldStorageConfig::create([
        'field_name' => self::FIELD_NAME,
        'entity_type' => 'node',
        'type' => 'entity_reference',
        'settings' => [
          'target_type' => 'commerce_product',
        ],
        'cardinality' => 1,
      ])->save();
    }

    // Create field instance if it doesn't exist.
    if (!FieldConfig::loadByName('node', $bundle, self::FIELD_NAME)) {
      FieldConfig::create([
        'field_name' => self::FIELD_NAME,
        'entity_type' => 'node',
        'bundle' => $bundle,
        'label' => 'Pay Publish Product',
        'settings' => [
          'handler' => 'default:commerce_product',
          'handler_settings' => [
            'target_bundles' => ['pay_publish' => 'pay_publish'],
            'auto_create' => FALSE,
          ],
        ],
        'required' => TRUE,
      ])->save();

    }

    // Set default published status to unpublished for this content type.
    $node_type = $this->entityTypeManager
      ->getStorage('node_type')
      ->load($bundle);

    if ($node_type) {
      $node_type->status = FALSE;
      $node_type->save();
    }
  }

  /**
   * Removes the commerce product reference field from a content type.
   */
  private function removePayPublishProductField(string $bundle): void {
    if ($field = FieldConfig::loadByName('node', $bundle, self::FIELD_NAME)) {
      $field->delete();
    }
  }

  /**
   * Adds or updates the Source Nid field on the commerce_order default bundle.
   *
   * @param array $target_bundles
   *   The node bundles to allow as reference targets.
   */
  private function addOrUpdateSourceNidFieldOnOrder(array $target_bundles): void {
    $field_name = 'field_source_nid';
    $entity_type = 'commerce_order_item';
    $bundle = 'default';

    // Create field storage if it doesn't exist.
    if (!FieldStorageConfig::loadByName($entity_type, $field_name)) {
      FieldStorageConfig::create([
        'field_name' => $field_name,
        'entity_type' => $entity_type,
        'type' => 'entity_reference',
        'settings' => [
          'target_type' => 'node',
        ],
        // Unlimited.
        'cardinality' => -1,
      ])->save();
    }

    // Create or update field instance.
    $field_config = FieldConfig::loadByName($entity_type, $bundle, $field_name);
    if (!$field_config) {
      $field_config = FieldConfig::create([
        'field_name' => $field_name,
        'entity_type' => $entity_type,
        'bundle' => $bundle,
        'label' => 'Source Nid',
        'settings' => [
          'handler' => 'default:node',
          'handler_settings' => [
            'target_bundles' => array_combine($target_bundles, $target_bundles),
            'auto_create' => FALSE,
          ],
        ],
        'required' => FALSE,
      ]);
      $field_config->save();
    }
    else {
      // Update allowed bundles if field already exists.
      $settings = $field_config->get('settings');
      $settings['handler_settings']['target_bundles'] = array_combine($target_bundles, $target_bundles);
      $field_config->set('settings', $settings);
      $field_config->save();
    }

    // Set widget to select list in form display.
    $entity_form_display = $this->entityTypeManager
      ->getStorage('entity_form_display')
      ->load($entity_type . '.' . $bundle . '.default');
    if ($entity_form_display) {
      // Check if the method exists to avoid compatibility issues.
      if (method_exists($entity_form_display, 'setComponent')) {
        $entity_form_display->setComponent($field_name, [
          'type' => 'options_select',
        ]);
      }
      else {
        $content = $entity_form_display->get('content');
        $content[$field_name] = [
          'type' => 'options_select',
        ];
        $entity_form_display->set('content', $content);
      }
      $entity_form_display->save();
    }

    // Set field to "Rendered entity" in view display.
    $entity_view_display = $this->entityTypeManager
      ->getStorage('entity_view_display')
      ->load($entity_type . '.' . $bundle . '.default');
    if ($entity_view_display) {
      if (method_exists($entity_view_display, 'setComponent')) {
        $entity_view_display->setComponent($field_name, [
          'type' => 'entity_reference_entity_view',
          'settings' => [
            'view_mode' => 'default',
            'link' => FALSE,
          ],
          'label' => 'above',
          'weight' => 0,
        ]);
      }
      else {
        $content = $entity_view_display->get('content');
        $content[$field_name] = [
          'type' => 'entity_reference_entity_view',
          'settings' => [
            'view_mode' => 'default',
            'link' => FALSE,
          ],
          'label' => 'above',
          'weight' => 0,
        ];
        $entity_view_display->set('content', $content);
      }
      $entity_view_display->save();
    }
  }

}
