<?php

declare(strict_types=1);

namespace Drupal\commerce_pay_publish\EventSubscriber;

use Drupal\commerce_order\Event\OrderEvent;
use Drupal\commerce_order\Event\OrderEvents;
use Drupal\commerce_pay_publish\Service\NotificationService;
use Drupal\commerce_pay_publish\Service\PayPublishPlanManager;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\node\Entity\Node;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Listens to order completion events to handle node publishing.
 */
class OrderCompleteSubscriber implements EventSubscriberInterface, ContainerInjectionInterface {

  /**
   * The PayPublishPlanManager service.
   *
   * @var \Drupal\commerce_pay_publish\Service\PayPublishPlanManager
   */
  protected PayPublishPlanManager $planManager;

  /**
   * The NotificationService for sending notifications.
   *
   * @var \Drupal\commerce_pay_publish\Service\NotificationService
   */
  protected NotificationService $notificationService;

  /**
   * Constructs a new OrderCompleteSubscriber object.
   *
   * @param \Drupal\commerce_pay_publish\Service\PayPublishPlanManager $planManager
   *   The PayPublishPlanManager service.
   * @param \Drupal\commerce_pay_publish\Service\NotificationService $notificationService
   *   The NotificationService for sending notifications.
   */
  public function __construct(PayPublishPlanManager $planManager, NotificationService $notificationService) {
    $this->planManager = $planManager;
    $this->notificationService = $notificationService;
  }

  /**
   * Creates an instance of the service via dependency injection.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('commerce_pay_publish.pay_publish_plan_manager'),
      $container->get('commerce_pay_publish.notification')

    );
  }

  /**
   * Returns the subscribed events.
   */
  public static function getSubscribedEvents() {
    return [
      OrderEvents::ORDER_PAID => 'onOrderPaid',
    ];
  }

  /**
   * Handles the order paid event to publish nodes and create plans.
   */
  public function onOrderPaid(OrderEvent $event) {
    $order = $event->getOrder();
    foreach ($order->getItems() as $order_item) {
      if ($order_item->hasField('field_source_nid') && !$order_item->get('field_source_nid')->isEmpty()) {
        $nid = $order_item->get('field_source_nid')->target_id;
        $node = Node::load($nid);
        if ($node && !$node->isPublished()) {
          $node->setPublished(TRUE);
          $node->save();
          $this->notificationService->sendPublishNotification($node);
        }
        // Create pay_publish_purchased_plan record.
        $plan = $this->planManager->createPurchasedPlan($order, $order_item);
        if ($plan) {
          $this->planManager->addPlanUsage($node);
        }
      } else {
        $this->planManager->createPurchasedPlan($order, $order_item);
      }
    }

  }

}
