<?php

declare(strict_types=1);

namespace Drupal\commerce_pay_publish\Service;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\Entity\OrderItemInterface;
use Drupal\commerce_pay_publish\Entity\PayPublishPlan;
use Drupal\commerce_pay_publish\Entity\PayPublishPlanUsage;
use Drupal\commerce_pay_publish\ProductBundle\PayPublishProductBundleInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\node\Entity\Node;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Manages Pay to Publish plans and their associations with users.
 */
class PayPublishPlanManager {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * Constructs a PayPublishPlanManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, AccountProxyInterface $currentUser, ConfigFactoryInterface $configFactory) {
    $this->entityTypeManager = $entityTypeManager;
    $this->currentUser = $currentUser;
    $this->config = $configFactory->get('commerce_pay_publish.settings');
  }

  /**
   * Creates a pay_publish_plan entity record from order and order item.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   * @param \Drupal\commerce_order\Entity\OrderItemInterface $order_item
   *   The order item entity.
   *
   * @return \Drupal\commerce_pay_publish\Entity\PayPublishPlan|null
   *   The created entity or NULL on failure.
   */
  public function createPurchasedPlan(OrderInterface $order, OrderItemInterface $order_item): ?PayPublishPlan {
    $product_variation = $order_item->getPurchasedEntity();
    $product = NULL;
    if ($product_variation instanceof FieldableEntityInterface && $product_variation->hasField('product_id') && !$product_variation->get('product_id')->isEmpty()) {
      $product_id = $product_variation->get('product_id')->target_id;
      $product = $this->entityTypeManager->getStorage('commerce_product')->load($product_id);
    }

    if (!$product instanceof PayPublishProductBundleInterface) {
      return NULL;
    }

    $publish_limit = $product->getBundleLimit();
    $expire = $product->getProductExpire();

    $expiry_date = NULL;
    $status = 1;
    if (!empty($expire['interval']) && !empty($expire['period'])) {
      $interval_spec = 'P' . $expire['interval'] . strtoupper(substr($expire['period'], 0, 1));
      try {
        $expiry = (new \DateTimeImmutable())->add(new \DateInterval($interval_spec));
        $expiry_date = $expiry->getTimestamp();
      }
      catch (\Exception $e) {
        $expiry_date = NULL;
      }
    }

    $quantity = $order_item->getQuantity();

    $values = [
      'order_id' => $order->id(),
      'order_item_id' => $order_item->id(),
      'product_id' => $product ? $product->id() : NULL,
      'uid' => $order->getCustomerId(),
      'purchased_date' => time(),
      'expiry_date' => $expiry_date,
      'publish_limit' => ($publish_limit * $quantity),
      'order_item_quantity' => $quantity,
      'used_nodes' => 0,
      'status' => $status,
    ];

    return $this->createPlan($values);
  }

  /**
   * Creates a pay_publish_plan entity record.
   *
   * @param array $values
   *   The values to create the plan with.
   *
   * @return \Drupal\commerce_pay_publish\Entity\PayPublishPlan
   *   The created entity.
   */
  public function createPlan(array $values): PayPublishPlan {
    /** @var \Drupal\commerce_pay_publish\Entity\PayPublishPlan $entity */
    $plan = $this->entityTypeManager
      ->getStorage('commerce_pay_publish_plan')
      ->create($values);
    $plan->save();
    return $plan;
  }

  /**
   * Gets the active purchased plan for a user.
   *
   * @param int $uid
   *   The user ID.
   *
   * @return \Drupal\commerce_pay_publish\Entity\PayPublishPlan|null
   *   The active plan entity or NULL if none found.
   */
  public function getActivePurchasedPlan(?int $uid = NULL): ?PayPublishPlan {
    $uid = $uid ?? (int) $this->currentUser->id();
    $storage = $this->entityTypeManager->getStorage('commerce_pay_publish_plan');
    $query = $storage->getQuery()
      ->condition('uid', $uid)
      ->condition('status', 1)
     // ->condition('used_nodes', 0, '<')
      ->condition('publish_limit', 0, '>')
      ->accessCheck(FALSE);
    $ids = $query->sort('expiry_date', 'ASC')->range(0, 1)->execute();
    if ($ids) {
      $entities = $storage->loadMultiple($ids);
      /** @var \Drupal\commerce_pay_publish\Entity\PayPublishPlan|null $entity */
      $entity = reset($entities) ?: NULL;
      return $entity;
    }
    // Check if trial/free plan is available.
    $trial_plan = $this->getTrialPlan();
    if ($trial_plan) {
      return $trial_plan;
    }
    return NULL;
  }

  /**
   * Gets the trial/free plan.
   *
   * @return \Drupal\commerce_pay_publish\Entity\PayPublishPlan|null
   *   The trial/free plan entity or NULL if none found.
   */
  public function getTrialPlan(): ?PayPublishPlan {
    $default_pay_publish_product = $this->config->get('default_pay_publish_product');
    if (!$default_pay_publish_product) {
      return NULL;
    }
    $uid = $this->currentUser->id();
    $storage = $this->entityTypeManager->getStorage('commerce_pay_publish_plan');
    $query = $storage->getQuery()
      ->condition('uid', $uid)
      ->condition('product_id', $default_pay_publish_product)
      ->accessCheck(FALSE);
    $ids = $query->sort('expiry_date', 'ASC')->range(0, 1)->execute();
    if (!$ids) {
       /** @var \Drupal\commerce_pay_publish\ProductBundle\PayPublishProductBundleInterface $product */
      $product = $this->entityTypeManager->getStorage('commerce_product')->load($default_pay_publish_product);
      if (!$product) {
        return NULL;
      }
      $publish_limit = $product->getBundleLimit();
      $expire = $product->getProductExpire();
      $expiry_date = NULL;

      if (!empty($expire['interval']) && !empty($expire['period'])) {
        $expiry_date = strtotime('+' . $expire['interval'] . ' ' . $expire['period']);
      }

      $values = [
        'order_id' => 0,
        'order_item_id' => 0,
        'uid' => $uid,
        'purchased_date' => time(),
        'product_id' => $default_pay_publish_product,
        'publish_limit' => $publish_limit,
        'expiry_date' => $expiry_date,
        'status' => 1,
        'used_nodes' => 0,
        'order_item_quantity' => 0,
      ];
      $plan = $this->createPlan($values);
      return $plan;
    }
    return NULL;
  }

  /**
   * Adds a plan usage record for a node, creating a pay_publish_plan entity.
   *
   * @param \Drupal\node\Entity\Node $node
   *   The node entity for which the plan usage is being created.
   * @param bool $resave
   *   Whether to resave the node after publishing.
   *
   * @return \Drupal\commerce_pay_publish\Entity\PayPublishPlanUsage|null
   *   The created plan usage entity or NULL if no active plan found.
   */
  public function addPlanUsage(Node $node, $resave = TRUE): ?PayPublishPlanUsage {
    $active_plan = $this->getActivePurchasedPlan();
    if (!$active_plan) {
      return NULL;
    }
    // Check if there are remaining credits.
    $remaining_credits = $active_plan->get('publish_limit')->value - $active_plan->get('used_nodes')->value;
    if ($remaining_credits <= 0) {
      return NULL;
    }

    // Load the related product entity from the plan.
    $product_id = $active_plan->get('product_id')->value ?? NULL;
    $product = $product_id ? $this->entityTypeManager->getStorage('commerce_product')->load($product_id) : NULL;

    $expiry_date = NULL;
    if ($product instanceof PayPublishProductBundleInterface) {
      $expire = $product->getExpirationDuration();
      if (!empty($expire['interval']) && !empty($expire['period'])) {
        $expiry_date = strtotime('+' . $expire['interval'] . ' ' . $expire['period']);
      }
    }

    $values = [
      'plan_id' => $active_plan->id(),
      'nid' => $node->id(),
      'publish_date' => time(),
      'expiry_date' => $expiry_date,
      'status' => 1,
    ];
    $entity = $this->entityTypeManager
      ->getStorage('commerce_pay_publish_plan_usage')
      ->create($values);
    $entity->save();

    // Publish the node.
    $node->setPublished();
    if ($resave) {
      $node->save();
    }

    $active_plan->addUsedNodes();
    $active_plan->save();
    return $entity;
  }

  /**
   * Checks if a user has an active plan for a given product.
   *
   * @param int|string $product_id
   *   The commerce_product entity ID.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   (optional) The user account. If not provided, uses the current user.
   *
   * @return bool
   *   TRUE if the user has an active plan for the product, FALSE otherwise.
   */
  public function userHasActivePlan(int $product_id, ?AccountInterface $account): bool {
    $account = $account ?: $this->currentUser;
    $storage = $this->entityTypeManager->getStorage('commerce_pay_publish_plan');
    $query = $storage->getQuery()
      ->condition('uid', $account->id())
      ->condition('product_id', $product_id)
      ->condition('publish_limit', 0, '>')
      ->condition('status', 1)
      ->accessCheck(FALSE);
    $ids = $query->execute();
    if (empty($ids)) {
      return FALSE;
    }
    $entities = $storage->loadMultiple($ids);
    foreach ($entities as $entity) {
      $remaining_credits = $entity->get('publish_limit')->value - $entity->get('used_nodes')->value;
      if ($remaining_credits > 0) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * Checks if a user has a plan for a given product.
   */
  public function shouldShowRelistButton(Node $node): bool {
    $config = \Drupal::config('commerce_pay_publish.settings');
    $enabled_types = $config->get('enabled_node_types') ?: [];
    if (!in_array($node->bundle(), $enabled_types, TRUE)) {
      return FALSE;
    }
    if ($node->isPublished()) {
      return FALSE;
    }

    // Check if there is a plan usage record for this node.
    $usage_storage = \Drupal::entityTypeManager()->getStorage('commerce_pay_publish_plan_usage');
    $usage_ids = $usage_storage->getQuery()
      ->condition('nid', $node->id())
      ->condition('status', 0)
      ->sort('publish_date', 'DESC')
      ->range(0, 1)
      ->accessCheck(FALSE)
      ->execute();
    $is_usage_ids  = !empty($usage_ids);

    if ($is_usage_ids) {
      // Check if user has an active plan.
      $has_active_plan = $this->getActivePurchasedPlan();

      // If user has an active plan, check if plan has enough credits.
      if (!$has_active_plan) {
        return TRUE;
      }
    }

    return $is_usage_ids;
  }

  /**
   * Gets the plan usage and associated plan for a given node.
   */
  public function getPlanUsageAndPlan(Node $node): ?array {
    $usage_storage = $this->entityTypeManager->getStorage('commerce_pay_publish_plan_usage');
    $plan_storage = $this->entityTypeManager->getStorage('commerce_pay_publish_plan');

    // Find the latest usage for this node.
    $usage_ids = $usage_storage->getQuery()
      ->condition('nid', $node->id())
      ->sort('publish_date', 'DESC')
      ->range(0, 1)
      ->accessCheck(FALSE)
      ->execute();

    if (empty($usage_ids)) {
      return NULL;
    }

    /** @var \Drupal\commerce_pay_publish\Entity\PayPublishPlanUsage $usage */
    $usage = $usage_storage->load(reset($usage_ids));
    if (!$usage) {
      return NULL;
    }

    $plan_id = $usage->getPlanId();
    if (!$plan_id) {
      return NULL;
    }

    /** @var \Drupal\commerce_pay_publish\Entity\PayPublishPlan $plan */
    $plan = $plan_storage->load($plan_id);

    return [
      'commerce_pay_publish_plan_usage' => $usage,
      'commerce_pay_publish_plan' => $plan,
      'node' => $node,
    ];
  }

  /**
   * Redirects to checkout with the relist product.
   *
   * @param \Drupal\node\Entity\Node $node
   *   The node entity to relist.
   */
  public function redirectCheckout(Node $node) {
    \Drupal::messenger()->addStatus(t('Redirecting to checkout to purchase a plan.'));
    $url = Url::fromRoute('commerce_pay_publish.add_to_cart', ['node' => $node->id()])->toString();
    $response = new RedirectResponse($url);
    $response->send();
    exit();
  }

}
