<?php

declare(strict_types=1);

namespace Drupal\commerce_pay_publish\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Handles relisting of nodes in the Commerce Pay to Publish system.
 */
class CommercePayPublishRelistController extends ControllerBase {

  /**
   * Relists a node by checking if the user has an active plan.
   */
  public function relist(NodeInterface $node) {
    $plan_manager = \Drupal::service('commerce_pay_publish.pay_publish_plan_manager');
    $notification_service = \Drupal::service('commerce_pay_publish.notification');

    // Check if user has an active plan and node is unpublished.
    if ($node->isPublished()) {
      $this->messenger()->addError($this->t('Relisting is not available. The content must be unpublished.'));
      $url = Url::fromRoute('entity.node.canonical', ['node' => $node->id()])->toString();
      return new RedirectResponse($url);
    }

    // If no active plan, show product selection form.
    if (!$plan_manager->getActivePurchasedPlan()) {
      // Get the latest plan usage to determine expiration date.
      $plan_usage_info = $plan_manager->getPlanUsageAndPlan($node);
      $expired_date = '';

      if ($plan_usage_info && isset($plan_usage_info['commerce_pay_publish_plan_usage'])) {
        $usage = $plan_usage_info['commerce_pay_publish_plan_usage'];
        $expiry_timestamp = $usage->get('expiry_date')->value;
        if ($expiry_timestamp) {
          $expired_date = \Drupal::service('date.formatter')->format($expiry_timestamp, 'custom', 'F j, Y');
        }
      }

      $build = [];
      $build['product_info'] = [
        '#type' => 'markup',
        '#markup' => '<h5>This Item Expired On: ' . $expired_date . '</h5>',
      ];
      $build['form'] = \Drupal::formBuilder()->getForm(
        'Drupal\commerce_pay_publish\Form\ProductSelectForm',
        $node
      );
      return $build;
    }

    // User has active plan, proceed with relisting.
    $plan_manager->addPlanUsage($node);
    $notification_service->sendRenewalNotification($node);
    $this->messenger()->addStatus($this->t('Your listing has been renewed.'));
    $url = Url::fromRoute('entity.node.canonical', ['node' => $node->id()])->toString();
    return new RedirectResponse($url);
  }

  /**
   * Checks if the current user has access to relist a node.
   */
  public function access(NodeInterface $node) {
    $plan_manager = \Drupal::service('commerce_pay_publish.pay_publish_plan_manager');

    // Check if node has an associated product for relisting.
    $has_product = $this->hasAssociatedProduct($node);
    if (!$has_product) {
      return AccessResult::forbidden();
    }

    // Check if the user has an active plan and if relisting is allowed.
    $should_show_relist = $plan_manager->shouldShowRelistButton($node);
    $has_active_plan = $plan_manager->getActivePurchasedPlan();

    return AccessResult::allowedIf($should_show_relist || !$has_active_plan);
  }

  /**
   * Checks if a node has an associated product for relisting.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to check.
   *
   * @return bool
   *   TRUE if the node has an associated product, FALSE otherwise.
   */
  protected function hasAssociatedProduct(NodeInterface $node) {
    // First try to get from the field_pay_publish_product field.
    if ($node->hasField('field_pay_publish_product') && !$node->get('field_pay_publish_product')->isEmpty()) {
      return TRUE;
    }

    // Fallback to the custom table if the field doesn't exist.
    $connection = \Drupal::database();
    $plan_id = $connection
      ->select('commerce_pay_publish_plan_usage', 'cp')
      ->fields('cp', ['plan_id'])
      ->condition('nid', $node->id())
      ->execute()
      ->fetchField();

    return !empty($plan_id);
  }

}
