<?php

namespace Drupal\commerce_pay_publish\Form;

use Drupal\commerce_product\Entity\ProductVariation;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Provides a form for publishing a product variation.
 */
class ProductSelectForm extends FormBase {

  /**
   * Returns the form ID.
   */
  public function getFormId() {
    return 'commerce_pay_publish_product_select_form';
  }

  /**
   * Builds the product selection form.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *
   * @return array
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = \Drupal::config('commerce_pay_publish.settings');
    $default_product = $config->get('default_pay_publish_product');
    $node_type_products = $config->get('node_type_products') ?? [];
    $enabled_types = $config->get('enabled_node_types') ?? [];

    $options = [];

    // Try to get node from route or query param.
    $node = \Drupal::routeMatch()->getParameter('node');
    if (!$node) {
      $nid = \Drupal::request()->query->get('nid');
      if ($nid && is_numeric($nid)) {
        $node = \Drupal\node\Entity\Node::load($nid);
      }
    }

    $product_ids = [];

    if ($node) {
      $form['node_id'] = [
        '#type' => 'hidden',
        '#value' => $node->id(),
      ];

      $bundle = $node->bundle();
      if (in_array($bundle, $enabled_types, true) && isset($node_type_products[$bundle])) {
        $product_ids = array_filter($node_type_products[$bundle]);
      }
    }

    // Load products: filtered or all active ones.
    $products = !empty($product_ids)
      ? \Drupal::entityTypeManager()->getStorage('commerce_product')->loadMultiple($product_ids)
      : \Drupal::entityTypeManager()->getStorage('commerce_product')->loadByProperties(['status' => 1]);

    foreach ($products as $product) {
      if ($product->id() == $default_product) {
        continue;
      }

      $variations = $product->getVariations();
      if (!empty($variations)) {
        $variation = reset($variations);
        $options[$variation->id()] = $product->label();
      }
    }

    $form['product_select'] = [
      '#type' => 'select',
      '#title' => $this->t('Select a Product'),
      '#options' => $options,
      '#empty_option' => $this->t('Select a Product'),
      '#required' => TRUE,
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add to Cart & Checkout'),
    ];

    return $form;
  }

  /**
   * Handles the form submission.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $variation_id = $form_state->getValue('product_select');
    $node_id = $form_state->getValue('node_id');

    if ($variation_id) {
      $variation = ProductVariation::load($variation_id);
      if ($variation) {
        $cart_provider = \Drupal::service('commerce_cart.cart_provider');
        $cart_manager = \Drupal::service('commerce_cart.cart_manager');
        $cart = $cart_provider->getCart('default');
        if (!$cart) {
          $cart = $cart_provider->createCart('default');
        }

        // Clear existing cart items.
        // foreach ($cart->getItems() as $item) {
        //   $cart->removeItem($item);
        // }
        // $cart->save();

        // Add the selected product to cart.
        $order_item = $cart_manager->addEntity($cart, $variation, 1);

        // Associate the node with the order item for payment completion.
        if ($node_id && $order_item) {
          $order_item->set('field_source_nid', $node_id);
          $order_item->save();
        }

        $response = new RedirectResponse(Url::fromRoute('commerce_checkout.form', ['commerce_order' => $cart->id()])->toString());
        $response->send();
        exit();
      }
    }
  }

}
