<?php

declare(strict_types=1);

namespace Drupal\commerce_pay_publish\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Handles cleanup tasks for the Commerce Pay to Publish module.
 */
class CommercePayPublishClean {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a CommercePayPublishClean object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger channel factory service.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, ConfigFactoryInterface $configFactory, LoggerChannelFactoryInterface $loggerFactory) {
    $this->entityTypeManager = $entityTypeManager;
    $this->configFactory = $configFactory;
    $this->logger = $loggerFactory->get('commerce_pay_publish');
  }

  /**
   * Executes cleanup tasks for the Pay to Publish system.
   *
   * @return void
   *   This method deletes all PayPublishPlan and PayPublishPlanUsage entities,
   */
  public function clean() : void {
    $this->deleteAllPlanEntities();
    $this->deleteProductType();
    $this->deleteVariationType();
    $this->deleteFieldStorages();
    $this->deleteFieldFromNodes('field_pay_publish_product');
    $this->deleteFieldFromOrderItems('field_source_nid');
    $this->deleteSettings();
  }

  /**
   * Deletes all PayPublishPlan and PayPublishPlanUsage entities.
   *
   * @return void
   *   This method will remove all plan entities from the system.
   */
  public function deleteAllPlanEntities(): void {
    // Delete all PayPublishPlan entities.
    try {
      $plan_storage = $this->entityTypeManager->getStorage('commerce_pay_publish_plan');
      $plan_ids = $plan_storage->getQuery()->accessCheck(FALSE)->execute();
      if (!empty($plan_ids)) {
        $plans = $plan_storage->loadMultiple($plan_ids);
        $plan_storage->delete($plans);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to delete PayPublishPlan entities: @message', ['@message' => $e->getMessage()]);
    }

    // Delete all PayPublishPlanUsage entities.
    try {
      $usage_storage = $this->entityTypeManager->getStorage('commerce_pay_publish_plan_usage');
      $usage_ids = $usage_storage->getQuery()->accessCheck(FALSE)->execute();
      if (!empty($usage_ids)) {
        $usages = $usage_storage->loadMultiple($usage_ids);
        $usage_storage->delete($usages);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to delete PayPublishPlanUsage entities: @message', ['@message' => $e->getMessage()]);
    }
  }

  /**
   * Deletes the custom Commerce product type used for Pay to Publish.
   *
   * @return void
   *   This method removes the 'pay_publish' product type from the system.
   */
  protected function deleteProductType() : void {
    try {
      $storage = $this->entityTypeManager->getStorage('commerce_product_type');
      $entity = $storage->load('pay_publish');
      if ($entity) {
        $entity->delete();
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to delete pay_publish product type: @message', ['@message' => $e->getMessage()]);
    }
  }

  /**
   * Deletes the custom Commerce product variation type.
   *
   * This method removes the 'pay_publish' variation type from the system.
   *
   * @return void
   *   This method deletes the 'pay_publish' variation type.
   */
  protected function deleteVariationType(): void {
    try {
      $storage = $this->entityTypeManager->getStorage('commerce_product_variation_type');
      $entity = $storage->load('pay_publish');
      if ($entity) {
        $entity->delete();
      }
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to delete pay_publish variation type: @message',
        [
          '@message' => $e->getMessage(),
        ]
      );
    }
  }

  /**
   * Deletes custom field storages used by the Pay to Publish module.
   *
   * @return void
   *   This method removes specific field storages related to the module.
   */
  protected function deleteFieldStorages() : void {
    try {
      $storage = $this->entityTypeManager->getStorage('field_storage_config');
      $fields = [
        'commerce_product.field_bundle_limit',
        'commerce_product.field_expiration_duration',
        'commerce_product.field_product_expire',
      ];
      foreach ($fields as $field_name) {
        $field = $storage->load($field_name);
        if ($field) {
          $field->delete();
        }
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to delete field storages: @message', ['@message' => $e->getMessage()]);
    }
  }

  /**
   * Deletes a specific field from all node bundles if it exists.
   *
   * @param string $field_name
   *   The name of the field to delete.
   *
   * @return void
   *   This method removes the specified field from all node bundles.
   */
  protected function deleteFieldFromNodes(string $field_name) : void {
    try {
      $storage = $this->entityTypeManager->getStorage('field_config');
      $bundles = \Drupal::service('entity_type.bundle.info')->getBundleInfo('node');
      foreach (array_keys($bundles) as $bundle) {
        $field = $storage->load('node.' . $bundle . '.' . $field_name);
        if ($field) {
          $field->delete();
        }
      }
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to delete @field from nodes: @message',
        [
          '@field' => $field_name,
          '@message' => $e->getMessage(),
        ]
      );
    }
  }

  /**
   * Deletes a specific field from all commerce order.
   *
   * @param string $field_name
   *   The name of the field to delete.
   *
   * @return void
   *   Removes the specified field from all commerce order.
   */
  protected function deleteFieldFromOrderItems(string $field_name) : void {
    try {
      $storage = $this->entityTypeManager->getStorage('field_config');
      $bundles = \Drupal::service('entity_type.bundle.info')->getBundleInfo('commerce_order_item');
      foreach (array_keys($bundles) as $bundle) {
        $field = $storage->load('commerce_order_item.' . $bundle . '.' . $field_name);
        if ($field) {
          $field->delete();
        }
      }
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to delete @field from order items: @message',
        [
          '@field' => $field_name,
          '@message' => $e->getMessage(),
        ]
      );
    }
  }

  /**
   * Deletes the configuration settings of the Pay to Publish module.
   *
   * @return void
   *   This method removes the module's configuration settings.
   */
  protected function deleteSettings() : void {
    try {
      $this->configFactory->getEditable('commerce_pay_publish.settings')->delete();
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to delete module settings: @message', ['@message' => $e->getMessage()]);
    }
  }

}
