<?php

declare(strict_types=1);

namespace Drupal\your_module\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\Entity\Node;
use Psr\Log\LoggerInterface;

/**
 * Manages logic related to node purchases via Commerce products.
 */
class NodePurchaseManager {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a NodePurchaseManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, LoggerInterface $logger) {
    $this->entityTypeManager = $entityTypeManager;
    $this->logger = $logger;
  }

  /**
   * Publishes a node after successful payment.
   *
   * @param int $nid
   *   The node ID.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user who made the purchase.
   *
   * @return bool
   *   TRUE on success, FALSE otherwise.
   */
  public function publishNodeAfterPurchase(int $nid, AccountInterface $account): bool {
    $node = $this->entityTypeManager->getStorage('node')->load($nid);
    if ($node instanceof Node && !$node->isPublished()) {
      $node->setPublished(TRUE);
      $node->save();
      $this->logger->info('Node @nid published after purchase by user @uid.', [
        '@nid' => $nid,
        '@uid' => $account->id(),
      ]);
      return TRUE;
    }
    return FALSE;
  }

}
