<?php

declare(strict_types=1);

namespace Drupal\commerce_pay_publish\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Utility\Token;
use Drupal\node\NodeInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for handling notifications in Commerce Pay Publish module.
 */
class NotificationService {

  use StringTranslationTrait;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The mail manager service.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected MailManagerInterface $mailManager;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected Token $token;

  /**
   * The PayPublishPlanManager service.
   *
   * @var \Drupal\commerce_pay_publish\Service\PayPublishPlanManager|null
   */
  protected PayPublishPlanManager $payPublishPlanManager;

  /**
   * Constructs a new NotificationService object.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    MailManagerInterface $mailManager,
    LoggerInterface $logger,
    EntityTypeManagerInterface $entity_type_manager,
    Token $token,
    PayPublishPlanManager $payPublishPlanManager,
  ) {
    $this->configFactory = $configFactory;
    $this->mailManager = $mailManager;
    $this->logger = $logger;
    $this->entityTypeManager = $entity_type_manager;
    $this->token = $token;
    $this->payPublishPlanManager = $payPublishPlanManager;
  }

  /**
   * Sends a notification when a node is published.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node that has been published.
   *
   * @return void
   *   This method sends an email notification to the node author.
   */
  public function sendPublishNotification(NodeInterface $node): void {
    if (!$this->areNotificationsEnabled()) {
      return;
    }

    $config = $this->configFactory->get('commerce_pay_publish.settings');
    $subject = $config->get('publish_email_subject') ?: 'Your content has been published';
    $body = $config->get('publish_email_body') ?: 'Your content "[node:title]" has been published.';

    $this->sendNotification($node, $subject, $body);
  }

  /**
   * Sends a notification when a node is unpublished.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node that has been unpublished.
   *
   * @return void
   *   This method sends an email notification to the node author.
   */
  public function sendExpireNotification(NodeInterface $node): void {
    if (!$this->areNotificationsEnabled()) {
      return;
    }

    $config = $this->configFactory->get('commerce_pay_publish.settings');
    $subject = $config->get('expire_email_subject') ?: 'Your content has expired';
    $body = $config->get('expire_email_body') ?: 'Your content "[node:title]" has expired.';

    $this->sendNotification($node, $subject, $body);
  }

  /**
   * Sends a notification when a node is relisted.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node that has been relisted.
   *
   * @return void
   *   This method sends an email notification to the node author.
   */
  public function sendRenewalNotification(NodeInterface $node): void {
    if (!$this->areNotificationsEnabled()) {
      return;
    }

    $config = $this->configFactory->get('commerce_pay_publish.settings');
    $subject = $config->get('renewal_email_subject') ?: 'Your content has been renewed';
    $body = $config->get('renewal_email_body') ?: 'Your content "[node:title]" has been renewed until [commerce_pay_publish_plan:expiry_date].';

    $this->sendNotification($node, $subject, $body);
  }

  /**
   * Sends a notification email to the node author.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node for which the notification is being sent.
   * @param string $subjectTemplate
   *   The email subject template.
   * @param string $bodyTemplate
   *   The email body template.
   *
   * @return void
   *   This method sends an email notification to the node author.
   */
  protected function sendNotification(NodeInterface $node, string $subjectTemplate, string $bodyTemplate): void {
    $author = $node->getOwner();
    if (!$author || !$author->getEmail()) {
      $this->logger->warning('Cannot send notification for node @nid: author has no email address.', [
        '@nid' => $node->id(),
      ]);
      return;
    }

    $data = $this->payPublishPlanManager->getPlanUsageAndPlan($node);
    if (!$data || !is_array($data)) {
      $this->logger->warning('Cannot send notification for node @nid: missing plan or usage data.', [
        '@nid' => $node->id(),
      ]);
      return;
    }
    $subject = $this->token->replace($subjectTemplate, $data, [], new BubbleableMetadata());
    $body = $this->token->replace($bodyTemplate, $data, [], new BubbleableMetadata());

    $params = [
      'subject' => $subject,
      'body' => $body,
      'node' => $node,
    ];

    $result = $this->mailManager->mail(
      'commerce_pay_publish',
      'notification',
      $author->getEmail(),
      $author->getPreferredLangcode(),
      $params
    );

    if (!empty($result['result'])) {
      $this->logger->info('Notification sent to @email for node @nid.', [
        '@email' => $author->getEmail(),
        '@nid' => $node->id(),
      ]);
    }
    else {
      $this->logger->error('Failed to send notification to @email for node @nid.', [
        '@email' => $author->getEmail(),
        '@nid' => $node->id(),
      ]);
    }
  }

  /**
   * Checks if notifications are enabled.
   *
   * @return bool
   *   TRUE if notifications are enabled, FALSE otherwise.
   */
  protected function areNotificationsEnabled(): bool {
    return (bool) $this->configFactory->get('commerce_pay_publish.settings')->get('notifications_enabled');
  }

}
