<?php

namespace Drupal\commerce_paynow\Controller;

use Drupal\commerce_payment\Entity\PaymentGatewayInterface;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Drupal\commerce_paynow\Service\PaynowLogger;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Handles Paynow webhook notifications.
 */
class PaynowWebhookController extends ControllerBase {

  /**
   * The Paynow logger service.
   *
   * @var \Drupal\commerce_paynow\Service\PaynowLogger
   */
  protected PaynowLogger $paynowLogger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->paynowLogger = $container->get('commerce_paynow.logger');
    return $instance;
  }

  /**
   * Processes incoming webhook notifications.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentGatewayInterface $commerce_payment_gateway
   *   The payment gateway entity.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function webhook(PaymentGatewayInterface $commerce_payment_gateway, Request $request) {
    /** @var \Drupal\commerce_paynow\Plugin\Commerce\PaymentGateway\Paynow $paymentGatewayPlugin */
    $paymentGatewayPlugin = $commerce_payment_gateway->getPlugin();

    $config = $paymentGatewayPlugin->getConfiguration();
    $this->paynowLogger->setEnableLogging($config['enable_logging'] ?? FALSE);

    $this->paynowLogger->info('Invoked Paynow webhook');

    try {
      if ($paymentGatewayPlugin->getPluginId() !== 'paynow') {
        return new Response('', 400);
      }
      return $paymentGatewayPlugin->onNotify($request) ?: new Response('', 400);
    }
    catch (\Exception $e) {
      $this->paynowLogger->error($this->t('Payment processing failed: @error', [
        '@error' => $e->getMessage(),
      ]));
      return new Response('', 400);
    }
  }

}
