<?php

namespace Drupal\commerce_paynow\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Attribute\CommercePaymentGateway;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsNotificationsInterface;
use Drupal\commerce_paynow\PluginForm\PaynowForm;
use Drupal\commerce_paynow\Service\PaynowLogger;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Paynow\Client as PaynowClient;
use Paynow\Environment as PaynowEnvironment;
use Paynow\Exception\PaynowException;
use Paynow\Notification as PaynowNotification;
use Paynow\Service\Payment as PaynowPayment;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Provides the Paynow payment gateway.
 */
#[CommercePaymentGateway(
  id: "paynow",
  label: new TranslatableMarkup("Paynow (mBank)"),
  display_label: new TranslatableMarkup("Paynow"),
  forms: [
    "offsite-payment" => PaynowForm::class,
  ],
  requires_billing_information: FALSE,
)]
class Paynow extends OffsitePaymentGatewayBase implements SupportsNotificationsInterface {

  /**
   * The Paynow SDK client.
   *
   * @var \Paynow\Client
   */
  protected PaynowClient $paynowClient;

  /**
   * The Paynow logger.
   *
   * @var \Drupal\commerce_paynow\Service\PaynowLogger
   */
  protected PaynowLogger $paynowLogger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->paynowClient = $instance->initializePaynowClient();
    $instance->paynowLogger = $container->get('commerce_paynow.logger');
    $instance->paynowLogger->setEnableLogging($configuration['enable_logging'] ?? FALSE);
    return $instance;
  }

  /**
   * Initialize the Paynow SDK client.
   */
  protected function initializePaynowClient(): PaynowClient {
    $environment = $this->configuration['environment'] === 'production'
      ? PaynowEnvironment::PRODUCTION
      : PaynowEnvironment::SANDBOX;

    $applicationName = $this->configuration['application_name'] ?? 'Drupal Commerce Paynow/3.0';

    return new PaynowClient(
      $this->configuration['api_key'],
      $this->configuration['signature_key'],
      $environment,
      $applicationName
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'api_key' => '',
      'signature_key' => '',
      'environment' => 'sandbox',
      'application_name' => 'Drupal Commerce Paynow/3.0',
      'enable_logging' => FALSE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $formState) {
    $form = parent::buildConfigurationForm($form, $formState);

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#default_value' => $this->configuration['api_key'],
      '#required' => TRUE,
      '#description' => $this->t('Your Paynow API key from the merchant panel.'),
    ];

    $form['signature_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Signature Key'),
      '#default_value' => $this->configuration['signature_key'],
      '#required' => TRUE,
      '#description' => $this->t('Your Paynow signature key from the merchant panel.'),
    ];

    $form['application_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Application Name'),
      '#description' => $this->t('Application name for User-Agent headers.'),
      '#default_value' => $this->configuration['application_name'],
    ];

    $form['environment'] = [
      '#type' => 'select',
      '#title' => $this->t('Environment'),
      '#default_value' => $this->configuration['environment'],
      '#options' => [
        'sandbox' => $this->t('Sandbox (https://api.sandbox.paynow.pl)'),
        'production' => $this->t('Production (https://api.paynow.pl)'),
      ],
      '#description' => $this->t('Select the environment for API calls.'),
    ];

    $form['enable_logging'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable API logging'),
      '#default_value' => $this->configuration['enable_logging'],
      '#description' => $this->t('Log API requests and responses for debugging.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['api_key'] = $values['api_key'];
      $this->configuration['signature_key'] = $values['signature_key'];
      $this->configuration['application_name'] = $values['application_name'];
      $this->configuration['environment'] = $values['environment'];
      $this->configuration['enable_logging'] = $values['enable_logging'];

      $this->paynowClient = $this->initializePaynowClient();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    $paymentId = $request->query->get('paymentId');
    $paymentStatus = $request->query->get('paymentStatus');

    if (!$paymentId || !$paymentStatus) {
      throw new PaymentGatewayException('Missing payment parameters in return URL.');
    }

    $paymentStorage = $this->entityTypeManager->getStorage('commerce_payment');
    $payments = $paymentStorage->loadByProperties([
      'remote_id' => $paymentId,
      'order_id' => $order->id(),
    ]);

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = reset($payments);

    try {
      $paymentService = new PaynowPayment($this->paynowClient);
      $statusResponse = $paymentService->status($paymentId);

      if ($statusResponse->getStatus() === 'CONFIRMED') {
        if ($payment && $payment->getState()->getId() === 'new') {
          $payment->setState('completed');
          $payment->setRemoteState($statusResponse->getStatus());
          $payment->setAuthorizedTime($this->time->getRequestTime());
          $payment->save();
        }
        else {
          $payment = $paymentStorage->create([
            'state' => 'completed',
            'amount' => $order->getTotalPrice(),
            'payment_gateway' => $this->parentEntity->id(),
            'order_id' => $order->id(),
            'remote_id' => $paymentId,
            'remote_state' => $statusResponse->getStatus(),
            'authorized' => $this->time->getRequestTime(),
          ]);
          $payment->save();
        }

        $this->messenger()->addMessage($this->t('Payment completed successfully.'));
      }
      elseif (in_array($statusResponse->getStatus(), ['REJECTED', 'ERROR', 'EXPIRED', 'ABANDONED'])) {
        throw new PaymentGatewayException('Payment was not successful: ' . $statusResponse->getStatus());
      }
    }
    catch (PaynowException $e) {
      $this->paynowLogger->error('Payment status check failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      throw new PaymentGatewayException('Error retrieving payment status: ' . $e->getMessage());
    }

  }

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request): ?Response {
    $payload = $request->getContent();
    $headers = $this->getRequestHeaders($request);

    try {
      $this->paynowLogger->debug('Paynow notification received: ' . $payload);

      new PaynowNotification($this->configuration['signature_key'], $payload, $headers);

      $data = Json::decode($payload);

      if (json_last_error() !== JSON_ERROR_NONE) {
        $this->paynowLogger->error('Invalid Paynow data received.');
        return NULL;
      }

      $paymentId = $data['paymentId'] ?? NULL;
      $status = $data['status'] ?? NULL;
      $externalId = $data['externalId'] ?? NULL;

      if (!$paymentId || !$status || !$externalId) {
        $this->paynowLogger->error('Missing required fields.');
        return NULL;
      }

      $notificationStatus = $this->processNotification($paymentId, $status, $externalId);

      return $notificationStatus ? new JsonResponse() : NULL;
    }
    catch (\Exception $e) {
      $this->paynowLogger->error('Paynow notification failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Process the payment notification with Paynow API status values.
   *
   * @param string $paymentId
   *   The payment ID.
   * @param string $status
   *   The payment status.
   * @param string $externalId
   *   The external ID (order number).
   */
  protected function processNotification(string $paymentId, string $status, string $externalId): bool {
    $orderStorage = $this->entityTypeManager->getStorage('commerce_order');
    $orders = $orderStorage->loadByProperties(['order_number' => $externalId]);

    if (empty($orders)) {
      $this->paynowLogger->error('Order not found for external ID: @id', [
        '@id' => $externalId,
      ]);
      return FALSE;
    }

    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = reset($orders);
    $paymentStorage = $this->entityTypeManager->getStorage('commerce_payment');

    switch ($status) {
      case 'CONFIRMED':
        $existingPayments = $paymentStorage->loadByProperties([
          'remote_id' => $paymentId,
          'order_id' => $order->id(),
        ]);

        if (empty($existingPayments)) {
          $payment = $paymentStorage->create([
            'state' => 'completed',
            'amount' => $order->getTotalPrice(),
            'payment_gateway' => $this->parentEntity->id(),
            'order_id' => $order->id(),
            'remote_id' => $paymentId,
            'remote_state' => $status,
          ]);
        }
        else {
          /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
          $payment = reset($existingPayments);
          if ($payment->getState()->getId() !== 'new') {
            $this->paynowLogger->warning('Payment already exists and is not in new state.');
            return FALSE;
          }
          $payment->setState('completed');
          $payment->setRemoteState($status);
          $payment->setAuthorizedTime($this->time->getRequestTime());
        }
        $payment->save();
        break;

      case 'REJECTED':
      case 'ERROR':
      case 'EXPIRED':
      case 'ABANDONED':
        /** @var \Drupal\commerce_payment\Entity\PaymentInterface[] $existingPayments */
        $existingPayments = $paymentStorage->loadByProperties([
          'remote_id' => $paymentId,
          'order_id' => $order->id(),
        ]);

        if (!empty($existingPayments)) {
          /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
          $payment = reset($existingPayments);
          $payment->setState('authorization_voided');
          $payment->setRemoteState($status);
          $payment->save();
        }
        break;

      case 'PENDING':
      case 'NEW':
        break;
    }

    return TRUE;
  }

  /**
   * Extract headers from the request for SDK notification verification.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return array
   *   Array of headers.
   */
  protected function getRequestHeaders(Request $request): array {
    $headers = [];
    foreach ($request->headers->all() as $key => $values) {
      $headers[$key] = is_array($values) ? reset($values) : $values;
    }
    return $headers;
  }

  /**
   * Get the Paynow client instance.
   *
   * @return \Paynow\Client
   *   The Paynow client.
   */
  public function getPaynowClient(): PaynowClient {
    return $this->paynowClient;
  }

}
