<?php

namespace Drupal\commerce_paynow\PluginForm;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm;
use Drupal\commerce_paynow\Service\OrderNumber;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Paynow\Service\Payment as PaynowPayment;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\commerce_paynow\Service\PaynowLogger;

/**
 * Form for Paynow payment gateway.
 */
class PaynowForm extends PaymentOffsiteForm implements ContainerInjectionInterface {

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected OrderNumber $orderNumber,
    protected ConfigFactoryInterface $configFactory,
    protected PaynowLogger $paynowLogger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('commerce_paynow.order_number'),
      $container->get('config.factory'),
      $container->get('commerce_paynow.logger'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $formState) {
    $form = parent::buildConfigurationForm($form, $formState);

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->entity;
    $order = $payment->getOrder();

    /** @var \Drupal\commerce_paynow\Plugin\Commerce\PaymentGateway\Paynow $paymentGatewayPlugin */
    $paymentGatewayPlugin = $payment->getPaymentGateway()->getPlugin();

    $config = $paymentGatewayPlugin->getConfiguration();
    $this->paynowLogger->setEnableLogging($config['enable_logging'] ?? FALSE);

    $this->paynowLogger->info('Starting Paynow payment authorization for order @order', [
      '@order' => $order->id(),
    ]);

    try {
      $paymentData = $this->buildPaymentData($payment, $order);
      $paynowClient = $paymentGatewayPlugin->getPaynowClient();
      $paymentService = new PaynowPayment($paynowClient);
      $idempotencyKey = $this->generateIdempotencyKey($order);

      $result = $paymentService->authorize($paymentData, $idempotencyKey);

      $payment->setRemoteId($result->getPaymentId());
      $payment->save();

      if ($result->getRedirectUrl()) {
        return $this->buildRedirectForm($form, $formState, $result->getRedirectUrl(), [], 'get');
      }
      else {
        $this->paynowLogger->error('No redirect URL received from Paynow API for order @order', [
          '@order' => $order->id(),
        ]);
        throw new \Exception('No redirect URL received from Paynow API.');
      }
    }
    catch (\Exception $e) {
      $this->paynowLogger->error('Paynow payment authorization failed for order @order: @error', [
        '@order' => $order->id(),
        '@error' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

  /**
   * Builds the payment data array for Paynow API.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentInterface $payment
   *   The payment entity.
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   *
   * @return array
   *   The payment data array
   */
  protected function buildPaymentData(PaymentInterface $payment, OrderInterface $order): array {
    $amount = $payment->getAmount();
    $billingProfile = $order->getBillingProfile();

    /** @var \Drupal\address\AddressInterface $billingAddress */
    $billingAddress = $billingProfile ? $billingProfile->get('address')->first() : NULL;

    // Ensure the order number is set following configured pattern.
    $this->orderNumber->setOrderNumber($order);
    $order->save();

    // Generate unique external ID.
    $externalId = $order->getOrderNumber() ?: $order->id();

    $data = [
      'amount' => ((float) $amount->getNumber()) * 100,
      'currency' => 'PLN',
      'externalId' => $externalId,
      'description' => $this->t('Order @order_number from @site_name', [
        '@order_number' => $externalId,
        '@site_name' => $this->configFactory->get('system.site')->get('name'),
      ])->render(),
      'buyer' => [
        'email' => $order->getEmail(),
      ],
    ];

    if ($billingAddress) {
      $data['buyer']['firstName'] = $billingAddress->getGivenName();
      $data['buyer']['lastName'] = $billingAddress->getFamilyName();

      $data['buyer']['address'] = [
        'billing' => [
          'street' => $billingAddress->getAddressLine1(),
          'houseNumber' => $billingAddress->getAddressLine2() ?: '1',
          'zipcode' => $billingAddress->getPostalCode(),
          'city' => $billingAddress->getLocality(),
          'country' => $billingAddress->getCountryCode(),
        ],
      ];
    }

    $data['orderItems'] = [];
    foreach ($order->getItems() as $orderItem) {
      $purchasedEntity = $orderItem->getPurchasedEntity();

      $data['orderItems'][] = [
        'name' => $purchasedEntity ? $purchasedEntity->label() : $orderItem->getTitle(),
        'category' => 'physical',
        'quantity' => (int) $orderItem->getQuantity(),
        'price' => ((float) $amount->getNumber()) * 100,
      ];
    }

    $data['continueUrl'] = Url::fromRoute('commerce_payment.checkout.return', [
      'commerce_order' => $order->id(),
      'step' => 'payment',
    ], ['absolute' => TRUE])->toString();

    return $data;
  }

  /**
   * Generate a unique idempotency key for the payment.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   *
   * @return string
   *   The idempotency key.
   */
  protected function generateIdempotencyKey(OrderInterface $order): string {
    $externalId = $order->getOrderNumber() ?: $order->id();
    return uniqid($externalId . '_', TRUE);
  }

}
