<?php

namespace Drupal\commerce_paynow\Service;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Psr\Log\LoggerInterface;
use Psr\Log\LogLevel;

/**
 * Logger for commerce_paynow channel based on gateway config.
 */
class PaynowLogger {

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Whether logging is enabled.
   *
   * @var bool
   */
  protected bool $enableLogging = FALSE;

  /**
   * Constructs the Paynow logger.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   * @param bool $enableLogging
   *   Whether logging is enabled.
   */
  public function __construct(LoggerChannelFactoryInterface $loggerFactory, bool $enableLogging = FALSE) {
    $this->logger = $loggerFactory->get('commerce_paynow');
    $this->enableLogging = $enableLogging;
  }

  /**
   * Set enableLogging flag.
   *
   * @param bool $enableLogging
   *   Whether logging should be enabled.
   */
  public function setEnableLogging(bool $enableLogging) {
    $this->enableLogging = $enableLogging;
  }

  /**
   * Logs a message if logging is enabled.
   *
   * @param string $level
   *   Log level (e.g., 'error', 'warning', 'info', 'debug').
   * @param string $message
   *   The log message.
   * @param array $context
   *   Context array.
   * @param bool $alwaysEnabled
   *   Whether to log this message regardless of the enableLogging flag.
   */
  public function log(string $level, string $message, array $context = [], bool $alwaysEnabled = FALSE) {
    if ($this->enableLogging || $alwaysEnabled) {
      $this->logger->log($level, $message, $context);
    }
  }

  /**
   * Logs an error message.
   */
  public function error($message, array $context = [], bool $alwaysEnabled = FALSE) {
    $this->log(LogLevel::ERROR, $message, $context, $alwaysEnabled);
  }

  /**
   * Logs a warning message.
   */
  public function warning($message, array $context = [], bool $alwaysEnabled = FALSE) {
    $this->log(LogLevel::WARNING, $message, $context, $alwaysEnabled);
  }

  /**
   * Logs an info message.
   */
  public function info($message, array $context = [], bool $alwaysEnabled = FALSE) {
    $this->log(LogLevel::INFO, $message, $context, $alwaysEnabled);
  }

  /**
   * Logs a debug message.
   */
  public function debug($message, array $context = [], bool $alwaysEnabled = FALSE) {
    $this->log(LogLevel::DEBUG, $message, $context, $alwaysEnabled);
  }

  /**
   * Logs an emergency message.
   */
  public function emergency($message, array $context = [], bool $alwaysEnabled = FALSE) {
    $this->log(LogLevel::EMERGENCY, $message, $context, $alwaysEnabled);
  }

  /**
   * Logs an alert message.
   */
  public function alert($message, array $context = [], bool $alwaysEnabled = FALSE) {
    $this->log(LogLevel::ALERT, $message, $context, $alwaysEnabled);
  }

  /**
   * Logs a critical message.
   */
  public function critical($message, array $context = [], bool $alwaysEnabled = FALSE) {
    $this->log(LogLevel::CRITICAL, $message, $context, $alwaysEnabled);
  }

  /**
   * Logs a notice message.
   */
  public function notice($message, array $context = [], bool $alwaysEnabled = FALSE) {
    $this->log(LogLevel::NOTICE, $message, $context, $alwaysEnabled);
  }

}
