<?php

namespace Drupal\Tests\commerce_paynow\Functional;

use Drupal\commerce_order\Entity\Order;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Tests\commerce\Functional\CommerceBrowserTestBase;
use Drupal\Tests\commerce_paynow\PaynowTestsTrait;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\IgnoreDeprecations;

/**
 * Functional tests for the Paynow payment gateway.
 */
#[Group('commerce_paynow')]
#[IgnoreDeprecations]
class PaynowFunctionalTest extends CommerceBrowserTestBase {

  use PaynowTestsTrait;

  /**
   * The payment gateway.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $gateway;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'commerce',
    'commerce_cart',
    'commerce_checkout',
    'commerce_number_pattern',
    'commerce_order',
    'commerce_payment',
    'commerce_price',
    'commerce_product',
    'commerce_store',
    'commerce_paynow',
    'entity',
    'entity_reference_revisions',
    'field',
    'filter',
    'inline_entity_form',
    'language',
    'options',
    'path',
    'path_alias',
    'profile',
    'state_machine',
    'system',
    'text',
    'user',
    'views',
  ];

  /**
   * {@inheritdoc}
   */
  protected function getAdministratorPermissions() {
    return array_merge([
      'administer commerce_order',
      'administer commerce_payment_gateway',
      'administer commerce_payment',
    ], parent::getAdministratorPermissions());
  }

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->gateway = $this->createTestPaymentGateway();

    $this->createTestProduct($this->store);

    /** @var \Drupal\Core\Entity\Display\EntityFormDisplayInterface $customer_form_display */
    $customer_form_display = EntityFormDisplay::load('profile.customer.default');
    $address_component = $customer_form_display->getComponent('address');
    $address_component['settings']['default_country'] = 'US';
    $customer_form_display->setComponent('address', $address_component);
    $customer_form_display->save();
  }

  /**
   * Tests the payment gateway configuration form.
   */
  public function testPaymentGatewayConfigurationForm() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/commerce/config/payment-gateways');

    $this->assertSession()->pageTextContains('Paynow Test');

    $this->clickLink('Edit');

    // Verify configuration form fields.
    $this->assertSession()->fieldExists('configuration[paynow][api_key]');
    $this->assertSession()->fieldExists('configuration[paynow][signature_key]');
    $this->assertSession()->fieldExists('configuration[paynow][environment]');
    $this->assertSession()->fieldExists('configuration[paynow][application_name]');

    // Test form validation.
    $edit = [
      'configuration[paynow][api_key]' => '',
      'configuration[paynow][signature_key]' => '',
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('API Key field is required');
  }

  /**
   * Tests the checkout process with Paynow gateway.
   */
  public function testCheckoutProcess() {
    $this->drupalGet('/product/1');
    $this->submitForm([], 'Add to cart');

    $this->drupalGet('/cart');
    $this->submitForm([], 'Checkout');

    $this->getSession()->getPage()->fillField('payment_information[billing_information][address][0][address][given_name]', 'John');
    $this->getSession()->getPage()->fillField('payment_information[billing_information][address][0][address][family_name]', 'Doe');
    $this->getSession()->getPage()->fillField('payment_information[billing_information][address][0][address][address_line1]', 'Test Street 123');
    $this->getSession()->getPage()->fillField('payment_information[billing_information][address][0][address][locality]', 'Anchorage');
    $this->getSession()->getPage()->fillField('payment_information[billing_information][address][0][address][administrative_area]', 'AK');
    $this->getSession()->getPage()->fillField('payment_information[billing_information][address][0][address][postal_code]', '99599');
    $this->submitForm([], 'Continue to review');

    $this->assertSession()->pageTextContains('Paynow');
    $this->assertSession()->pageTextContains('John');
    $this->assertSession()->pageTextContains('Doe');
    $this->assertSession()->pageTextContains('Test Street 123');
    $this->assertSession()->pageTextContains('Anchorage');

    $this->submitForm([], 'Pay and complete purchase');

    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests webhook endpoint failures.
   */
  public function testWebhookEndpointFailures() {
    $webhook_url = '/commerce-paynow/webhook/' . $this->gateway->id();

    // Test GET request (should not be allowed)
    $this->drupalGet($webhook_url);
    $this->assertSession()->statusCodeEquals(405);

    // Test POST request with invalid data.
    /** @var \GuzzleHttp\Client $client */
    $client = $this->getHttpClient();
    $absolute_url = $this->buildUrl($webhook_url);
    $response = $client->post($absolute_url, [
      'body' => '{"invalid":"data"}',
      'headers' => [
        'Content-Type' => 'application/json',
      ],
      'http_errors' => FALSE,
    ]);
    $this->assertEquals(400, $response->getStatusCode());
  }

  /**
   * Tests payment return URL handling.
   */
  public function testPaymentReturnUrl() {
    $order = Order::create([
      'type' => 'default',
      'store_id' => $this->store->id(),
      'uid' => $this->loggedInUser->id(),
      'mail' => $this->loggedInUser->getEmail(),
      'state' => 'draft',
    ]);
    $order->save();

    $return_url = '/checkout/' . $order->id() . '/payment/return';

    $this->drupalGet($return_url, [
      'query' => [
        'paymentId' => 'test-payment-123',
        'paymentStatus' => 'CONFIRMED',
      ],
    ]);

    $this->assertSession()->statusCodeNotEquals(500);
  }

  /**
   * Tests payment gateway display in admin.
   */
  public function testPaymentGatewayAdminDisplay() {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('/admin/commerce/config/payment-gateways');
    $this->assertSession()->pageTextContains('Paynow Test');
    $this->assertSession()->pageTextContains('Enabled');

    $this->clickLink('Edit');
    $this->assertSession()->pageTextContains('API Key');
    $this->assertSession()->pageTextContains('Signature Key');
    $this->assertSession()->pageTextContains('Environment');

    $environment_field = $this->getSession()->getPage()->findField('configuration[paynow][environment]');
    $this->assertNotNull($environment_field);

    $options = $environment_field->findAll('css', 'option');
    $option_values = [];
    foreach ($options as $option) {
      $option_values[] = $option->getValue();
    }

    $this->assertContains('sandbox', $option_values);
    $this->assertContains('production', $option_values);
  }

}
