<?php

namespace Drupal\Tests\commerce_paynow\Kernel\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_payment\Entity\Payment;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_store\StoreCreationTrait;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\commerce_paynow\PaynowTestsTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\IgnoreDeprecations;
use Symfony\Component\HttpFoundation\Request;

/**
 * Kernel tests for the Paynow payment gateway plugin.
 */
#[Group('commerce_paynow')]
#[IgnoreDeprecations]
class PaynowKernelTest extends KernelTestBase {

  use StoreCreationTrait;
  use UserCreationTrait;
  use PaynowTestsTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'commerce',
    'commerce_number_pattern',
    'commerce_order',
    'commerce_payment',
    'commerce_price',
    'commerce_product',
    'commerce_store',
    'commerce_paynow',
    'entity',
    'entity_reference_revisions',
    'field',
    'filter',
    'inline_entity_form',
    'options',
    'path',
    'path_alias',
    'profile',
    'state_machine',
    'system',
    'text',
    'user',
    'views',
  ];

  /**
   * The payment gateway.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $gateway;

  /**
   * The payment gateway plugin.
   *
   * @var \Drupal\commerce_paynow\Plugin\Commerce\PaymentGateway\Paynow
   */
  protected $plugin;

  /**
   * A test order.
   *
   * @var \Drupal\commerce_order\Entity\OrderInterface
   */
  protected $order;

  /**
   * A test payment.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentInterface
   */
  protected $payment;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('commerce_product');
    $this->installEntitySchema('commerce_product_variation');
    $this->installEntitySchema('commerce_order');
    $this->installEntitySchema('commerce_order_item');
    $this->installEntitySchema('commerce_payment');
    $this->installEntitySchema('commerce_payment_gateway');
    $this->installEntitySchema('commerce_store');
    $this->installEntitySchema('profile');
    $this->installConfig(['commerce_product', 'commerce_order', 'system']);
    $this->installConfig('commerce_payment');
    $this->installConfig('commerce_store');
    $this->installSchema('commerce_number_pattern', ['commerce_number_pattern_sequence']);
    $this->installConfig('commerce_paynow');

    $store = $this->createStore();
    $this->gateway = $this->createTestPaymentGateway();
    $this->plugin = $this->gateway->getPlugin();
    $this->order = $this->createTestOrder($store);

    $this->payment = Payment::create([
      'type' => 'payment_default',
      'payment_gateway' => $this->gateway->id(),
      'order_id' => $this->order->id(),
      'amount' => $this->order->getTotalPrice(),
      'currency_code' => $this->order->getTotalPrice()->getCurrencyCode(),
      'state' => 'new',
    ]);
    $this->payment->save();
  }

  /**
   * Tests payment gateway configuration.
   */
  public function testGatewayConfiguration() {
    $this->assertEquals('paynow', $this->gateway->getPluginId());
    $this->assertTrue($this->gateway->status());

    $config = $this->plugin->getConfiguration();
    $this->assertEquals('sandbox', $config['environment']);
    $this->assertEquals(self::$apiKey, $config['api_key']);
  }

  /**
   * Tests payment return with missing parameters.
   */
  public function testOnReturnMissingParameters() {
    $request = new Request();

    $this->expectException(PaymentGatewayException::class);
    $this->expectExceptionMessage('Missing payment parameters in return URL.');

    $this->plugin->onReturn($this->order, $request);
  }

  /**
   * Tests webhook notification validation.
   */
  public function testWebhookNotificationValidation() {
    $validPayload = json_encode([
      'paymentId' => 'test-payment-123',
      'status' => 'CONFIRMED',
      'externalId' => '12345',
    ]);

    $signature = base64_encode(hash_hmac(
      'sha256',
      $validPayload,
      self::$signatureKey,
      TRUE
    ));

    $request = new Request([], [], [], [], [], [], $validPayload);
    $request->headers->set('Signature', $signature);

    $this->assertNotNull($this->plugin->onNotify($request));
  }

  /**
   * Tests webhook notification with invalid signature.
   */
  public function testWebhookInvalidSignature() {
    $payload = json_encode([
      'paymentId' => 'test-payment-123',
      'status' => 'CONFIRMED',
      'externalId' => '12345',
    ]);

    $request = new Request([], [], [], [], [], [], $payload);
    $request->headers->set('Signature', 'invalid-signature');

    $this->assertNull($this->plugin->onNotify($request));
  }

  /**
   * Tests payment status processing for different statuses.
   */
  #[DataProvider('paymentStatusProvider')]
  public function testPaymentStatusProcessing($status, $expectedState) {
    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('processNotification');
    $method->setAccessible(TRUE);

    try {
      $method->invokeArgs($this->plugin, [
        'test-payment-123',
        $status,
        $this->order->getOrderNumber() ?: $this->order->id(),
      ]);

      $this->assertContainsEquals($expectedState, $this->payment->getState(), "Payment state should be '$expectedState' for status '$status'");
    }
    catch (\Exception $e) {
      $this->assertInstanceOf(\Exception::class, $e);
    }
  }

  /**
   * Data provider for payment status tests.
   */
  public static function paymentStatusProvider() {
    return [
      ['CONFIRMED', 'completed'],
      ['REJECTED', 'authorization_voided'],
      ['ERROR', 'authorization_voided'],
      ['EXPIRED', 'authorization_voided'],
      ['ABANDONED', 'authorization_voided'],
    ];
  }

}
