<?php

namespace Drupal\Tests\commerce_paynow;

use Drupal\commerce_order\Entity\Order;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\Entity\OrderItem;
use Drupal\commerce_payment\Entity\PaymentGateway;
use Drupal\commerce_price\Price;
use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_product\Entity\ProductVariation;
use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\commerce_store\Entity\StoreInterface;
use Drupal\commerce_store\StoreCreationTrait;
use Drupal\profile\Entity\Profile;
use Drupal\Tests\user\Traits\UserCreationTrait;

/**
 * Trait for Paynow tests.
 */
trait PaynowTestsTrait {

  use StoreCreationTrait;
  use UserCreationTrait;

  /**
   * The API key for testing Paynow.
   *
   * @var string
   */
  public static $apiKey = '97a55694-5478-43b5-b406-fb49ebfdd2b5';

  /**
   * The signature key for Paynow.
   *
   * @var string
   */
  public static $signatureKey = 'b305b996-bca5-4404-a0b7-2ccea3d2b64b';

  /**
   * The environment for testing Paynow.
   *
   * @var string
   */
  public static $environment = 'sandbox';

  /**
   * The application name for testing Paynow.
   *
   * @var string
   */
  public static $applicationName = 'Drupal Commerce Paynow';

  /**
   * Creates a payment gateway for testing.
   *
   * @return \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   *   The created payment gateway.
   */
  protected function createTestPaymentGateway() {
    $gateway = PaymentGateway::create([
      'id' => 'paynow_test',
      'label' => 'Paynow Test',
      'plugin' => 'paynow',
    ]);
    $gateway->getPlugin()->setConfiguration([
      'api_key' => self::$apiKey,
      'signature_key' => self::$signatureKey,
      'environment' => self::$environment,
      'application_name' => self::$applicationName,
    ]);
    $gateway->save();

    return $gateway;
  }

  /**
   * Creates a test product.
   *
   * @param \Drupal\commerce_store\Entity\StoreInterface|null $store
   *   The store.
   *
   * @return \Drupal\commerce_product\Entity\ProductVariationInterface
   *   The test product.
   */
  protected function createTestProduct(?StoreInterface $store = NULL): ProductVariationInterface {
    if (!$store) {
      $store = $this->createStore();
    }

    $variation = ProductVariation::create([
      'type' => 'default',
      'sku' => 'test-product',
      'title' => 'Test Product',
      'price' => new Price('29.99', 'USD'),
    ]);
    $variation->save();

    $product = Product::create([
      'type' => 'default',
      'title' => 'Test Product',
      'variations' => [$variation],
      'stores' => [$store],
    ]);
    $product->save();

    $variation = ProductVariation::load($variation->id());

    return $variation;
  }

  /**
   * Creates a test order with items.
   *
   * @param \Drupal\commerce_store\Entity\StoreInterface|null $store
   *   The store.
   *
   * @return \Drupal\commerce_order\Entity\OrderInterface
   *   The test order.
   */
  protected function createTestOrder(?StoreInterface $store = NULL): OrderInterface {
    $variation = $this->createTestProduct($store);

    $orderItem = OrderItem::create([
      'type' => 'default',
      'purchased_entity' => $variation,
      'quantity' => 2,
      'unit_price' => $variation->getPrice(),
    ]);
    $orderItem->save();

    $profile = Profile::create([
      'type' => 'customer',
      'address' => [
        'country_code' => 'US',
        'administrative_area' => 'AK',
        'locality' => 'Anchorage',
        'postal_code' => '00000',
        'address_line1' => 'Test Street 123',
        'given_name' => 'John',
        'family_name' => 'Doe',
      ],
    ]);
    $profile->save();

    $order = Order::create([
      'type' => 'default',
      'store_id' => $store->id(),
      'order_items' => [$orderItem],
      'state' => 'draft',
      'mail' => 'fotobankier@gmail.com',
      'billing_profile' => $profile,
      'placed' => time(),
      'order_number' => '12345',
    ]);
    $order->save();

    return $order;
  }

}
