<?php

/*
 * This file is part of the commerce_paypay package.
 *
 * (c) Rémi SIMAER <rsimaer@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Drupal\commerce_paypay\Controller;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentGatewayInterface;
use Drupal\commerce_paypay\Enum\PayPayPaymentStatus;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for PayPay webhook callbacks.
 */
class PayPayWebhookController extends ControllerBase {

  /**
   * The logger channel name.
   */
  private const LOGGER_CHANNEL = 'commerce_paypay';

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * Handles webhook notifications from PayPay.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function handle(Request $request): JsonResponse {
    try {
      $content = $request->getContent();
      $data = json_decode($content, TRUE);

      if (json_last_error() !== JSON_ERROR_NONE) {
        $this->getLogger(self::LOGGER_CHANNEL)->error('Invalid JSON in webhook: @error', [
          '@error' => json_last_error_msg(),
        ]);
        return new JsonResponse(['status' => 'error', 'message' => 'Invalid JSON'], Response::HTTP_BAD_REQUEST);
      }

      // Extract payment information.
      $merchant_payment_id = $data['merchantPaymentId'] ?? NULL;
      $payment_id = $data['paymentId'] ?? NULL;
      $status = $data['status'] ?? NULL;

      if (!$merchant_payment_id || !$payment_id || !$status) {
        $this->getLogger(self::LOGGER_CHANNEL)->error('Missing required webhook data');
        return new JsonResponse(['status' => 'error', 'message' => 'Missing required data'], Response::HTTP_BAD_REQUEST);
      }

      // Find the order.
      $order = $this->findOrderByMerchantPaymentId($merchant_payment_id);
      
      if (!$order) {
        $this->getLogger(self::LOGGER_CHANNEL)->warning('Order not found for merchant payment ID: @id', [
          '@id' => $merchant_payment_id,
        ]);
        return new JsonResponse(['status' => 'error', 'message' => 'Order not found'], Response::HTTP_NOT_FOUND);
      }

      // Get the payment gateway.
      $payment_gateway = $this->getPayPayGateway();
      
      if (!$payment_gateway) {
        $this->getLogger(self::LOGGER_CHANNEL)->error('PayPay payment gateway not found for order: @order_id', [
          '@order_id' => $order->id(),
        ]);
        return new JsonResponse(['status' => 'error', 'message' => 'Payment gateway not found'], Response::HTTP_NOT_FOUND);
      }

      // Process the webhook based on status.
      $this->processWebhook($order, $payment_gateway, $payment_id, $status);

      return new JsonResponse(['status' => 'success'], Response::HTTP_OK);
    }
    catch (\Exception $e) {
      $this->getLogger(self::LOGGER_CHANNEL)->error('Webhook processing error: @message', [
        '@message' => $e->getMessage(),
      ]);
      return new JsonResponse(['status' => 'error', 'message' => 'Internal error'], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
  }

  /**
   * Find order by merchant payment ID.
   *
   * @param string $merchantPaymentId
   *   The merchant payment ID.
   *
   * @return \Drupal\commerce_order\Entity\OrderInterface|null
   *   The order or NULL.
   */
  private function findOrderByMerchantPaymentId(string $merchantPaymentId): ?OrderInterface {
    // Try to extract order ID from merchant payment ID (format: ORDER_ID-TIMESTAMP).
    if (preg_match('/^(\d+)-\d+$/', $merchantPaymentId, $matches)) {
      $order_id = $matches[1];
      $order = $this->entityTypeManager->getStorage('commerce_order')->load($order_id);
      
      if ($order instanceof OrderInterface) {
        // Verify the stored merchant payment ID matches.
        $stored_id = $order->getData('paypay_merchant_payment_id');
        if ($stored_id === $merchantPaymentId) {
          return $order;
        }
      }
    }

    // Fallback: search all orders by data field.
    // Note: This is less efficient but necessary since data field is not directly queryable.
    $order_storage = $this->entityTypeManager->getStorage('commerce_order');
    $order_ids = $order_storage->getQuery()
      ->accessCheck(FALSE)
      ->execute();

    foreach ($order_ids as $order_id) {
      $order = $order_storage->load($order_id);
      if ($order instanceof OrderInterface && $order->getData('paypay_merchant_payment_id') === $merchantPaymentId) {
        return $order;
      }
    }

    return NULL;
  }

  /**
   * Get PayPay payment gateway for order.
   *
   * @return \Drupal\commerce_payment\Entity\PaymentGatewayInterface|null
   *   The payment gateway or NULL.
   */
  private function getPayPayGateway(): ?PaymentGatewayInterface {
    $payment_gateway_storage = $this->entityTypeManager->getStorage('commerce_payment_gateway');
    $payment_gateways = $payment_gateway_storage->loadByProperties(['plugin' => 'paypay']);

    foreach ($payment_gateways as $payment_gateway) {
      if ($payment_gateway instanceof PaymentGatewayInterface) {
        return $payment_gateway;
      }
    }

    return NULL;
  }

  /**
   * Process webhook data.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   * @param \Drupal\commerce_payment\Entity\PaymentGatewayInterface $paymentGateway
   *   The payment gateway.
   * @param string $paymentId
   *   The PayPay payment ID.
   * @param string $status
   *   The payment status.
   */
  private function processWebhook(
    OrderInterface $order,
    PaymentGatewayInterface $paymentGateway,
    string $paymentId,
    string $status
  ): void {
    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
    $payment_status = PayPayPaymentStatus::tryFrom($status);

    // Check if payment already exists.
    $existing_payments = $payment_storage->loadByProperties([
      'remote_id' => $paymentId,
      'order_id' => $order->id(),
    ]);

    if (!empty($existing_payments)) {
      // Update existing payment.
      $payment = reset($existing_payments);
      $payment->setRemoteState($status);
      
      switch ($payment_status) {
        case PayPayPaymentStatus::COMPLETED:
          $payment->setState('completed');
          break;

        case PayPayPaymentStatus::AUTHORIZED:
          $payment->setState('authorization');
          break;

        case PayPayPaymentStatus::FAILED:
        case PayPayPaymentStatus::CANCELED:
        case PayPayPaymentStatus::EXPIRED:
          $payment->setState('voided');
          break;
      }
      
      $payment->save();
      
      $this->getLogger(self::LOGGER_CHANNEL)->info('Updated payment @payment_id for order @order_id with status @status', [
        '@payment_id' => $paymentId,
        '@order_id' => $order->id(),
        '@status' => $status,
      ]);
    }
    else {
      // Create new payment if it doesn't exist and status is valid.
      if (in_array($payment_status, [PayPayPaymentStatus::COMPLETED, PayPayPaymentStatus::AUTHORIZED])) {
        $payment_state = $payment_status === PayPayPaymentStatus::COMPLETED ? 'completed' : 'authorization';
        
        /** @var \Drupal\commerce_paypay\Plugin\Commerce\PaymentGateway\PayPayPaymentGateway $gateway_plugin */
        $gateway_plugin = $paymentGateway->getPlugin();
        $gateway_plugin->createPayment($payment_storage, $order, $paymentGateway->id(), $paymentId, $status, $payment_state);
        
        $this->getLogger(self::LOGGER_CHANNEL)->info('Created payment @payment_id for order @order_id with status @status', [
          '@payment_id' => $paymentId,
          '@order_id' => $order->id(),
          '@status' => $status,
        ]);
      }
    }
  }

}
