<?php

/*
 * This file is part of the commerce_paypay package.
 *
 * (c) Rémi SIMAER <rsimaer@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Drupal\commerce_paypay\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsAuthorizationsInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsRefundsInterface;
use Drupal\commerce_paypay\Enum\PayPayConfigKey;
use Drupal\commerce_paypay\Enum\PayPayPaymentStatus;
use Drupal\commerce_paypay\Service\PayPayClientService;
use Drupal\commerce_price\Price;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the PayPay payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "paypay",
 *   label = "PayPay",
 *   display_label = "PayPay",
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_paypay\PluginForm\PayPayPaymentForm",
 *   },
 *   modes = {
 *     "authorize" = @Translation("Authorization (reserve funds, capture later)"),
 *     "capture" = @Translation("Capture (charge immediately)"),
 *   },
 * )
 */
class PayPayPaymentGateway extends OffsitePaymentGatewayBase implements SupportsAuthorizationsInterface, SupportsRefundsInterface {

  /**
   * The PayPay client service.
   */
  protected PayPayClientService $payPayClient;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->payPayClient = $container->get('commerce_paypay.client');
    return $instance;
  }

  /**
   * Initialize PayPay client with configuration.
   */
  private function initializePayPayClient(): void {
    $this->payPayClient->initializeClient(
      $this->configuration[PayPayConfigKey::API_KEY->value],
      $this->configuration[PayPayConfigKey::API_SECRET->value],
      $this->configuration[PayPayConfigKey::MERCHANT_ID->value],
      $this->configuration[PayPayConfigKey::PRODUCTION_MODE->value]
    );
  }

  /**
   * Get the PayPay client service with initialized configuration.
   *
   * @return \Drupal\commerce_paypay\Service\PayPayClientService
   *   The initialized PayPay client service.
   */
  public function getPayPayClient(): PayPayClientService {
    $this->initializePayPayClient();
    return $this->payPayClient;
  }

  /**
   * Generate a unique merchant transaction ID.
   *
   * @param int $orderId
   *   The order ID.
   * @param string $type
   *   The transaction type (refund, capture, void).
   *
   * @return string
   *   The unique merchant transaction ID.
   */
  private function generateMerchantId(int $orderId, string $type): string {
    return $orderId . '-' . $type . '-' . time();
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      PayPayConfigKey::API_KEY->value => '',
      PayPayConfigKey::API_SECRET->value => '',
      PayPayConfigKey::MERCHANT_ID->value => '',
      PayPayConfigKey::PRODUCTION_MODE->value => FALSE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form[PayPayConfigKey::API_KEY->value] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('Your PayPay API key.'),
      '#default_value' => $this->configuration[PayPayConfigKey::API_KEY->value],
      '#required' => TRUE,
    ];

    $form[PayPayConfigKey::API_SECRET->value] = [
      '#type' => 'password',
      '#title' => $this->t('API Secret'),
      '#description' => $this->t('Your PayPay API secret.'),
      '#default_value' => $this->configuration[PayPayConfigKey::API_SECRET->value],
      '#required' => TRUE,
    ];

    $form[PayPayConfigKey::MERCHANT_ID->value] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant ID'),
      '#description' => $this->t('Your PayPay merchant ID.'),
      '#default_value' => $this->configuration[PayPayConfigKey::MERCHANT_ID->value],
      '#required' => TRUE,
    ];

    $form[PayPayConfigKey::PRODUCTION_MODE->value] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Production mode'),
      '#description' => $this->t('Enable this to use the production environment. Disable for sandbox testing.'),
      '#default_value' => $this->configuration[PayPayConfigKey::PRODUCTION_MODE->value],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);
    $values = $form_state->getValue($form['#parents']);
    $this->configuration[PayPayConfigKey::API_KEY->value] = $values[PayPayConfigKey::API_KEY->value];
    $this->configuration[PayPayConfigKey::API_SECRET->value] = $values[PayPayConfigKey::API_SECRET->value];
    $this->configuration[PayPayConfigKey::MERCHANT_ID->value] = $values[PayPayConfigKey::MERCHANT_ID->value];
    $this->configuration[PayPayConfigKey::PRODUCTION_MODE->value] = $values[PayPayConfigKey::PRODUCTION_MODE->value];
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request): void {
    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
    
    // Get merchant payment ID from order data.
    $merchant_payment_id = $order->getData('paypay_merchant_payment_id');
    if (empty($merchant_payment_id)) {
      throw new PaymentGatewayException('PayPay payment ID not found on order. Payment was not properly initialized.');
    }

    // Initialize PayPay client.
    $this->initializePayPayClient();

    // Get payment details from PayPay.
    $response = $this->payPayClient->getPaymentDetails($merchant_payment_id);

    if ($response['resultInfo']['code'] !== 'SUCCESS') {
      $error_msg = $response['resultInfo']['message'] ?? 'Unknown error';
      $error_code = $response['resultInfo']['codeId'] ?? 'N/A';
      throw new PaymentGatewayException(sprintf('Payment verification failed: %s [%s]. Merchant Payment ID: %s', $error_msg, $error_code, $merchant_payment_id));
    }

    $payment_data = $response['data'];
    $status = PayPayPaymentStatus::tryFrom($payment_data['status']);

    // Create or update payment based on status.
    switch ($status) {
      case PayPayPaymentStatus::COMPLETED:
        $this->createPayment($payment_storage, $order, $this->parentEntity->id(), $payment_data['paymentId'], $payment_data['status'], 'completed');
        $this->messenger()->addStatus($this->t('Payment completed successfully.'));
        break;

      case PayPayPaymentStatus::AUTHORIZED:
        $this->createPayment($payment_storage, $order, $this->parentEntity->id(), $payment_data['paymentId'], $payment_data['status'], 'authorization');
        $this->messenger()->addStatus($this->t('Payment authorized successfully.'));
        break;

      case PayPayPaymentStatus::FAILED:
      case PayPayPaymentStatus::CANCELED:
      case PayPayPaymentStatus::EXPIRED:
        throw new PaymentGatewayException('Payment was not successful: ' . $payment_data['status']);

      default:
        throw new PaymentGatewayException('Payment status is unclear: ' . ($payment_data['status'] ?? 'Unknown'));
    }
  }

  /**
   * Create a payment entity.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $payment_storage
   *   The payment storage.
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   * @param string $payment_gateway_id
   *   The payment gateway ID.
   * @param string $remote_id
   *   The remote payment ID.
   * @param string $remote_state
   *   The remote payment state.
   * @param string $state
   *   The payment state.
   */
  public function createPayment($payment_storage, OrderInterface $order, string $payment_gateway_id, string $remote_id, string $remote_state, string $state): void {
    $payment = $payment_storage->create([
      'state' => $state,
      'amount' => $order->getTotalPrice(),
      'payment_gateway' => $payment_gateway_id,
      'order_id' => $order->id(),
      'remote_id' => $remote_id,
      'remote_state' => $remote_state,
    ]);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function onCancel(OrderInterface $order, Request $request): void {
    $this->messenger()->addWarning($this->t('Payment was cancelled.'));
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, ?Price $amount = NULL) {
    $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
    
    $amount = $amount ?: $payment->getAmount();
    $merchant_refund_id = $this->generateMerchantId($payment->getOrderId(), 'refund');

    // Initialize PayPay client.
    $this->initializePayPayClient();

    // Perform refund.
    $response = $this->payPayClient->refundPayment(
      $merchant_refund_id,
      $payment->getRemoteId(),
      [
        'amount' => (int) $amount->getNumber(),
        'currency' => $amount->getCurrencyCode(),
      ],
      'Refund for order ' . $payment->getOrderId()
    );

    if ($response['resultInfo']['code'] !== 'SUCCESS') {
      throw new PaymentGatewayException('Refund failed: ' . ($response['resultInfo']['message'] ?? 'Unknown error'));
    }

    $old_refunded_amount = $payment->getRefundedAmount();
    $new_refunded_amount = $old_refunded_amount->add($amount);
    
    if ($new_refunded_amount->lessThan($payment->getAmount())) {
      $payment->setState('partially_refunded');
    }
    else {
      $payment->setState('refunded');
    }
    
    $payment->setRefundedAmount($new_refunded_amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function capturePayment(PaymentInterface $payment, ?Price $amount = NULL) {
    $this->assertPaymentState($payment, ['authorization']);
    
    $amount = $amount ?: $payment->getAmount();
    $merchant_capture_id = $this->generateMerchantId($payment->getOrderId(), 'capture');

    // Initialize PayPay client.
    $this->initializePayPayClient();

    // Get the merchant payment ID from the order data.
    $order = $payment->getOrder();
    $merchant_payment_id = $order->getData('paypay_merchant_payment_id');
    if (empty($merchant_payment_id)) {
      throw new PaymentGatewayException('Cannot capture payment: merchant payment ID not found in order data.');
    }

    // Capture the payment.
    $response = $this->payPayClient->capturePaymentAuth(
      $merchant_payment_id,
      $merchant_capture_id,
      [
        'amount' => (int) $amount->getNumber(),
        'currency' => $amount->getCurrencyCode(),
      ],
      'Capture for order ' . $payment->getOrderId()
    );

    if ($response['resultInfo']['code'] !== 'SUCCESS') {
      throw new PaymentGatewayException('Capture failed: ' . ($response['resultInfo']['message'] ?? 'Unknown error'));
    }

    $payment->setState('completed');
    $payment->setAmount($amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function voidPayment(PaymentInterface $payment): void {
    $this->assertPaymentState($payment, ['authorization']);

    // Initialize PayPay client.
    $this->initializePayPayClient();

    $merchant_revert_id = $this->generateMerchantId($payment->getOrderId(), 'void');

    // Revert the authorization.
    $response = $this->payPayClient->revertAuth(
      $merchant_revert_id,
      $payment->getRemoteId(),
      'Void authorization for order ' . $payment->getOrderId()
    );

    if ($response['resultInfo']['code'] !== 'SUCCESS') {
      throw new PaymentGatewayException('Void failed: ' . ($response['resultInfo']['message'] ?? 'Unknown error'));
    }

    $payment->setState('authorization_voided');
    $payment->save();
  }

}
