<?php

/*
 * This file is part of the commerce_paypay package.
 *
 * (c) Rémi SIMAER <rsimaer@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Drupal\commerce_paypay\PluginForm;

use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Provides the offsite payment form for PayPay.
 */
class PayPayPaymentForm extends PaymentOffsiteForm {

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->entity;
    $order = $payment->getOrder();
    
    /** @var \Drupal\commerce_paypay\Plugin\Commerce\PaymentGateway\PayPayPaymentGateway $payment_gateway_plugin */
    $payment_gateway_plugin = $payment->getPaymentGateway()->getPlugin();

    // Get PayPay client service from the gateway plugin.
    $payPayClient = $payment_gateway_plugin->getPayPayClient();

    // Generate merchant payment ID.
    $merchant_payment_id = $order->id() . '-' . time();
    
    // Store merchant payment ID on order for later reference using data field.
    $order->setData('paypay_merchant_payment_id', $merchant_payment_id);
    $order->save();

    // Prepare order items.
    $order_items = [];
    foreach ($order->getItems() as $order_item) {
      $order_items[] = [
        'name' => $order_item->getTitle(),
        'quantity' => (int) $order_item->getQuantity(),
        'unit_price' => [
          'amount' => (int) $order_item->getUnitPrice()->getNumber(),
          'currency' => $order_item->getUnitPrice()->getCurrencyCode(),
        ],
      ];
    }

    // Prepare redirect URL.
    $return_url = Url::fromRoute('commerce_payment.checkout.return', [
      'commerce_order' => $order->id(),
      'step' => 'payment',
    ], ['absolute' => TRUE])->toString();

    // Get user agent if available.
    $user_agent = NULL;
    $request = \Drupal::request();
    if ($request->headers->has('User-Agent')) {
      $user_agent = $request->headers->get('User-Agent');
    }

    // Create QR code.
    $response = $payPayClient->createQrCode(
      $merchant_payment_id,
      [
        'amount' => (int) $payment->getAmount()->getNumber(),
        'currency' => $payment->getAmount()->getCurrencyCode(),
      ],
      $order_items,
      $return_url,
      'WEB_LINK',
      $payment_gateway_plugin->getMode() === 'authorize',
      'Order #' . $order->getOrderNumber(),
      $user_agent
    );

    $error_message = $this->t('Unable to create payment. Please try again or contact support.');

    if ($response['resultInfo']['code'] !== 'SUCCESS') {
      \Drupal::messenger()->addError($error_message);
      throw new \Exception('PayPay QR code creation failed: ' . ($response['resultInfo']['message'] ?? 'Unknown error'));
    }

    $payment_url = $response['data']['url'] ?? NULL;
    
    if (!$payment_url) {
      \Drupal::messenger()->addError($error_message);
      throw new \Exception('PayPay did not return a payment URL.');
    }

    // Redirect to PayPay.
    return $this->buildRedirectForm(
      $form,
      $form_state,
      $payment_url,
      [],
      'get'
    );
  }

}
