<?php

/*
 * This file is part of the commerce_paypay package.
 *
 * (c) Rémi SIMAER <rsimaer@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Drupal\commerce_paypay\Service;

use Drupal\Core\Logger\LoggerChannelInterface;
use PayPay\OpenPaymentAPI\Client;
use PayPay\OpenPaymentAPI\Models\CreateQrCodePayload;
use PayPay\OpenPaymentAPI\Models\OrderItem;
use PayPay\OpenPaymentAPI\Models\RefundPaymentPayload;
use PayPay\OpenPaymentAPI\Models\CapturePaymentAuthPayload;
use PayPay\OpenPaymentAPI\Models\RevertAuthPayload;

/**
 * Service for interacting with PayPay API.
 */
class PayPayClientService {

  /**
   * The PayPay client.
   */
  private ?Client $client = NULL;

  /**
   * Constructs a PayPayClientService object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   */
  public function __construct(
    private readonly LoggerChannelInterface $logger
  ) {}

  /**
   * Initialize the PayPay client.
   *
   * @param string $apiKey
   *   The API key.
   * @param string $apiSecret
   *   The API secret.
   * @param string $merchantId
   *   The merchant ID.
   * @param bool $productionMode
   *   Whether to use production mode.
   *
   * @return \PayPay\OpenPaymentAPI\Client
   *   The PayPay client.
   */
  public function initializeClient(string $apiKey, string $apiSecret, string $merchantId, bool $productionMode = FALSE): Client {
    if ($this->client === NULL) {
      $this->client = new Client([
        'API_KEY' => $apiKey,
        'API_SECRET' => $apiSecret,
        'MERCHANT_ID' => $merchantId,
      ], $productionMode);
    }
    return $this->client;
  }

  /**
   * Create a QR code for payment.
   *
   * @param string $merchantPaymentId
   *   The merchant payment ID.
   * @param array $amount
   *   The amount array with keys:
   *   - 'amount' (int): The payment amount in the smallest currency unit.
   *   - 'currency' (string): The currency code ('JPY').
   * @param array $orderItems
   *   Array of order items, each containing:
   *   - 'name' (string): Item name.
   *   - 'quantity' (int): Item quantity.
   *   - 'unit_price' (array): Price array with 'amount' and 'currency'.
   * @param string $redirectUrl
   *   The redirect URL after payment.
   * @param string $redirectType
   *   The redirect type (WEB_LINK or APP_DEEP_LINK).
   * @param bool $isAuthorization
   *   Whether this is a pre-authorization.
   * @param string|null $orderDescription
   *   Optional order description.
   * @param string|null $userAgent
   *   Optional user agent string.
   *
   * @return array
   *   The API response array with 'resultInfo' and 'data' keys.
   */
  public function createQrCode(
    string $merchantPaymentId,
    array $amount,
    array $orderItems,
    string $redirectUrl,
    string $redirectType = 'WEB_LINK',
    bool $isAuthorization = FALSE,
    ?string $orderDescription = NULL,
    ?string $userAgent = NULL,
  ): array {
    try {
      $payload = new CreateQrCodePayload();
      $payload->setMerchantPaymentId($merchantPaymentId);
      $payload->setRequestedAt();
      $payload->setCodeType('ORDER_QR');
      $payload->setAmount($amount);
      $payload->setRedirectType($redirectType);
      $payload->setRedirectUrl($redirectUrl);

      if ($isAuthorization) {
        $payload->setIsAuthorization($isAuthorization);
      }

      if ($orderDescription) {
        $payload->setOrderDescription($orderDescription);
      }

      if ($userAgent) {
        $payload->setUserAgent($userAgent);
      }

      // Convert order items to PayPay format.
      $paypayOrderItems = [];
      foreach ($orderItems as $item) {
        $orderItem = (new OrderItem())
          ->setName($item['name'])
          ->setQuantity($item['quantity'])
          ->setUnitPrice($item['unit_price']);
        $paypayOrderItems[] = $orderItem;
      }
      $payload->setOrderItems($paypayOrderItems);

      $response = $this->client->code->createQRCode($payload);

      if ($response['resultInfo']['code'] !== 'SUCCESS') {
        $this->logger->error('PayPay QR code creation failed: @message', [
          '@message' => $response['resultInfo']['message'] ?? 'Unknown error',
        ]);
      }

      return $response;
    }
    catch (\Exception $e) {
      $this->logger->error('PayPay QR code creation exception: @message', [
        '@message' => $e->getMessage(),
      ]);
      return [
        'resultInfo' => [
          'code' => 'EXCEPTION',
          'message' => $e->getMessage(),
        ],
      ];
    }
  }

  /**
   * Get payment details.
   *
   * @param string $merchantPaymentId
   *   The merchant payment ID.
   *
   * @return array
   *   The API response.
   */
  public function getPaymentDetails(string $merchantPaymentId): array {
    try {
      $response = $this->client->code->getPaymentDetails($merchantPaymentId);

      if ($response['resultInfo']['code'] !== 'SUCCESS') {
        $this->logger->warning('PayPay get payment details failed: @message', [
          '@message' => $response['resultInfo']['message'] ?? 'Unknown error',
        ]);
      }

      return $response;
    }
    catch (\Exception $e) {
      $this->logger->error('PayPay get payment details exception: @message', [
        '@message' => $e->getMessage(),
      ]);
      return [
        'resultInfo' => [
          'code' => 'EXCEPTION',
          'message' => $e->getMessage(),
        ],
      ];
    }
  }

  /**
   * Refund a payment.
   *
   * @param string $merchantRefundId
   *   The merchant refund ID.
   * @param string $paymentId
   *   The PayPay payment ID.
   * @param array $amount
   *   The amount array with keys:
   *   - 'amount' (int): The refund amount in the smallest currency unit.
   *   - 'currency' (string): The currency code ('JPY').
   * @param string|null $reason
   *   Optional refund reason.
   *
   * @return array
   *   The API response array with 'resultInfo' and 'data' keys.
   */
  public function refundPayment(string $merchantRefundId, string $paymentId, array $amount, ?string $reason = NULL): array {
    try {
      $payload = new RefundPaymentPayload();
      $payload->setMerchantRefundId($merchantRefundId);
      $payload->setPaymentId($paymentId);
      $payload->setAmount($amount);
      $payload->setRequestedAt();

      if ($reason) {
        $payload->setReason($reason);
      }

      $response = $this->client->refund->refundPayment($payload);

      if ($response['resultInfo']['code'] !== 'SUCCESS') {
        $this->logger->error('PayPay refund payment failed: @message', [
          '@message' => $response['resultInfo']['message'] ?? 'Unknown error',
        ]);
      }

      return $response;
    }
    catch (\Exception $e) {
      $this->logger->error('PayPay refund payment exception: @message', [
        '@message' => $e->getMessage(),
      ]);
      return [
        'resultInfo' => [
          'code' => 'EXCEPTION',
          'message' => $e->getMessage(),
        ],
      ];
    }
  }

  /**
   * Capture a payment authorization.
   *
   * @param string $merchantPaymentId
   *   The merchant payment ID.
   * @param string $merchantCaptureId
   *   The merchant capture ID.
   * @param array $amount
   *   The amount array with keys:
   *   - 'amount' (int): The capture amount in the smallest currency unit.
   *   - 'currency' (string): The currency code ('JPY').
   * @param string $orderDescription
   *   The order description.
   *
   * @return array
   *   The API response array with 'resultInfo' and 'data' keys.
   */
  public function capturePaymentAuth(string $merchantPaymentId, string $merchantCaptureId, array $amount, string $orderDescription): array {
    try {
      $payload = new CapturePaymentAuthPayload();
      $payload->setMerchantPaymentId($merchantPaymentId);
      $payload->setMerchantCaptureId($merchantCaptureId);
      $payload->setAmount($amount);
      $payload->setRequestedAt();
      $payload->setOrderDescription($orderDescription);

      $response = $this->client->payment->capturePaymentAuth($payload);

      if ($response['resultInfo']['code'] !== 'SUCCESS') {
        $this->logger->error('PayPay capture payment failed: @message', [
          '@message' => $response['resultInfo']['message'] ?? 'Unknown error',
        ]);
      }

      return $response;
    }
    catch (\Exception $e) {
      $this->logger->error('PayPay capture payment exception: @message', [
        '@message' => $e->getMessage(),
      ]);
      return [
        'resultInfo' => [
          'code' => 'EXCEPTION',
          'message' => $e->getMessage(),
        ],
      ];
    }
  }

  /**
   * Revert a payment authorization.
   *
   * @param string $merchantRevertId
   *   The merchant revert ID.
   * @param string $paymentId
   *   The PayPay payment ID.
   * @param string|null $reason
   *   Optional revert reason.
   *
   * @return array
   *   The API response.
   */
  public function revertAuth(string $merchantRevertId, string $paymentId, ?string $reason = NULL): array {
    try {
      $payload = new RevertAuthPayload();
      $payload->setMerchantRevertId($merchantRevertId);
      $payload->setPaymentId($paymentId);
      $payload->setRequestedAt();

      if ($reason) {
        $payload->setReason($reason);
      }

      $response = $this->client->payment->revertAuth($payload);

      if ($response['resultInfo']['code'] !== 'SUCCESS') {
        $this->logger->error('PayPay revert auth failed: @message', [
          '@message' => $response['resultInfo']['message'] ?? 'Unknown error',
        ]);
      }

      return $response;
    }
    catch (\Exception $e) {
      $this->logger->error('PayPay revert auth exception: @message', [
        '@message' => $e->getMessage(),
      ]);
      return [
        'resultInfo' => [
          'code' => 'EXCEPTION',
          'message' => $e->getMessage(),
        ],
      ];
    }
  }

}
