<?php

namespace Drupal\commerce_payumoney\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Database\Connection;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Psr\Log\LoggerInterface;

/**
 * Provides PayUMoney webhook handling.
 */
class PayUMoneyController extends ControllerBase {

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The logger instance.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new PayUMoneyController object.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger instance.
   */
  public function __construct(MessengerInterface $messenger, Connection $database, LoggerInterface $logger) {
    $this->messenger = $messenger;
    $this->database = $database;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('messenger'),
      $container->get('database'),
      $container->get('logger.factory')->get('commerce_payumoney')
    );
  }

  /**
   * Handles PayUMoney notification/webhook.
   */
  public function notify(Request $request) {
    $response_data = $request->request->all();

    // Log the incoming webhook data
    $this->logger->info('PayUMoney webhook received: @data', [
      '@data' => json_encode($response_data)
    ]);

    if (empty($response_data['txnid'])) {
      $this->logger->error('PayUMoney webhook missing transaction ID');
      return new Response('Missing transaction ID', 400);
    }

    try {
      $this->processPaymentResponse($response_data);
      return new Response('OK', 200);
    }
    catch (\Exception $e) {
      $this->logger->error('PayUMoney webhook processing failed: @error', [
        '@error' => $e->getMessage()
      ]);
      return new Response('Processing failed', 500);
    }
  }

  /**
   * Handles successful payment return.
   */
  public function success(Request $request) {
    $response_data = $request->isMethod('POST') ? $request->request->all() : $request->query->all();

    try {
      $result = $this->processPaymentResponse($response_data);

      if ($result['status'] === 'success') {
        $this->messenger->addMessage($this->t('Payment completed successfully. Transaction ID: @txnid', [
          '@txnid' => $response_data['txnid']
        ]));

        // Redirect to order confirmation or checkout completion
        if (!empty($result['order'])) {
          return $this->redirect('entity.commerce_order.canonical', [
            'commerce_order' => $result['order']->id()
          ]);
        }
      }
      else {
        $this->messenger->addError($this->t('Payment verification failed. Please try again.'));
      }
    }
    catch (\Exception $e) {
      $this->logger->error('PayUMoney success processing failed: @error', [
        '@error' => $e->getMessage()
      ]);
      $this->messenger->addError($this->t('Payment processing failed. Please contact support.'));
    }

    return $this->redirect('<front>');
  }

  /**
   * Handles failed payment return.
   */
  public function failure(Request $request) {
    $response_data = $request->isMethod('POST') ? $request->request->all() : $request->query->all();

    $this->logger->warning('PayUMoney payment failure: @data', [
      '@data' => json_encode($response_data)
    ]);

    if (!empty($response_data['txnid'])) {
      // Log the failed transaction
      $this->logTransaction($response_data, 'failed');

      $this->messenger->addError($this->t('Payment failed. Transaction ID: @txnid. Reason: @error', [
        '@txnid' => $response_data['txnid'],
        '@error' => $response_data['error_Message'] ?? 'Unknown error'
      ]));
    }
    else {
      $this->messenger->addError($this->t('Payment failed. Please try again.'));
    }

    return $this->redirect('<front>');
  }

  /**
   * Processes PayUMoney payment response.
   *
   * @param array $response_data
   *   The payment response data.
   *
   * @return array
   *   Processing result with status and order info.
   *
   * @throws \Exception
   */
  protected function processPaymentResponse(array $response_data) {
    if (empty($response_data['txnid'])) {
      throw new \Exception('Missing transaction ID in payment response');
    }

    $txnid = $response_data['txnid'];
    $status = strtolower($response_data['status'] ?? 'failed');
    $order_info = $this->extractOrderInfo($response_data);

    // Log the transaction
    $this->logTransaction($response_data, $status);

    if ($status === 'success') {
      // Verify the hash to ensure authenticity
      if ($this->verifyPaymentHash($response_data)) {
        // Update the Commerce payment
        if ($order_info['order'] && $order_info['payment']) {
          $this->updateCommercePayment($order_info['payment'], $response_data);
        }

        return [
          'status' => 'success',
          'order' => $order_info['order'],
          'payment' => $order_info['payment']
        ];
      }
      else {
        throw new \Exception('Payment hash verification failed');
      }
    }

    return [
      'status' => 'failed',
      'order' => $order_info['order'],
      'payment' => $order_info['payment']
    ];
  }

  /**
   * Extracts order and payment information from response.
   *
   * @param array $response_data
   *   The payment response data.
   *
   * @return array
   *   Array with order and payment entities.
   */
  protected function extractOrderInfo(array $response_data) {
    $order = NULL;
    $payment = NULL;

    // Extract order ID from productinfo or udf fields
    if (!empty($response_data['productinfo'])) {
      preg_match('/ProductorderID(\d+)/', $response_data['productinfo'], $matches);
      if (!empty($matches[1])) {
        $order_id = $matches[1];
        $order = \Drupal::entityTypeManager()
          ->getStorage('commerce_order')
          ->load($order_id);

        if ($order) {
          // Find the associated payment
          $payments = \Drupal::entityTypeManager()
            ->getStorage('commerce_payment')
            ->loadByProperties([
              'order_id' => $order_id
            ]);

          if (!empty($payments)) {
            $payment = reset($payments);
          }
        }
      }
    }

    return [
      'order' => $order,
      'payment' => $payment
    ];
  }

  /**
   * Verifies the PayUMoney payment hash.
   *
   * @param array $response_data
   *   The payment response data.
   *
   * @return bool
   *   TRUE if hash is valid, FALSE otherwise.
   */
  protected function verifyPaymentHash(array $response_data) {
    // This would need to get the gateway configuration
    // For now, return TRUE as a placeholder
    // In production, implement proper hash verification
    return TRUE;
  }

  /**
   * Updates the Commerce payment entity.
   *
   * @param object $payment
   *   The payment entity.
   * @param array $response_data
   *   The payment response data.
   */
  protected function updateCommercePayment($payment, array $response_data) {
    if ($response_data['status'] === 'success') {
      $payment->setState('completed');
      $payment->setRemoteId($response_data['txnid']);
      $payment->setRemoteState($response_data['status']);
      $payment->save();
    }
    else {
      $payment->setState('authorization_voided');
      $payment->setRemoteId($response_data['txnid']);
      $payment->setRemoteState($response_data['status']);
      $payment->save();
    }
  }

  /**
   * Logs transaction details to database.
   *
   * @param array $response_data
   *   The payment response data.
   * @param string $status
   *   The transaction status.
   */
  protected function logTransaction(array $response_data, $status) {
    try {
      $order_info = $this->extractOrderInfo($response_data);

      $this->database->insert('commerce_payumoney_transactions')
        ->fields([
          'txnid' => $response_data['txnid'],
          'order_id' => $order_info['order'] ? $order_info['order']->id() : 0,
          'payment_id' => $order_info['payment'] ? $order_info['payment']->id() : 0,
          'status' => $status,
          'amount' => floatval($response_data['amount'] ?? 0),
          'gateway_response' => json_encode($response_data),
          'created' => \Drupal::time()->getRequestTime(),
          'updated' => \Drupal::time()->getRequestTime(),
        ])
        ->execute();
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to log PayUMoney transaction: @error', [
        '@error' => $e->getMessage()
      ]);
    }
  }

}
