<?php

namespace Drupal\commerce_payumoney\PluginForm\PayUMoneyRedirect;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * Provides the Off-site payment form.
 *
 * Note: This form extends FormBase instead of PaymentOffsiteForm
 * to avoid dependency issues when Commerce is not installed.
 * When Commerce is properly installed, change the extends clause to:
 * extends \Drupal\commerce_payment\PluginForm\PaymentOffsiteForm
 */
class PaymentPayUMoneyForm extends FormBase {

  use StringTranslationTrait;

  const PAYUMONEY_API_TEST_URL = 'https://test.payu.in/_payment';
  const PAYUMONEY_API_URL = 'https://secure.payu.in/_payment';

  /**
   * The payment entity.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentInterface
   */
  protected $payment;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'commerce_payumoney_redirect_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // In a real Commerce implementation, the payment would be passed via entity context
    // For now, we'll get it from the route or build parameters
    $payment = $form_state->get('payment') ?? $this->payment;

    if (!$payment) {
      // Fallback: try to get from route context or request
      // This would need to be implemented based on how Commerce calls this form
      throw new \Exception('Payment entity not found in form state');
    }

    $redirect_method = 'post';
    /** @var \Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayInterface $payment_gateway_plugin */
    $payment_gateway_plugin = $payment->getPaymentGateway()->getPlugin();

    $order = $payment->getOrder();
    $configuration = $payment_gateway_plugin->getConfiguration();
    $key = $configuration['pkey'];
    $salt = $configuration['psalt'];
    $mode = $payment_gateway_plugin->getMode();

    $service_provider = ($mode == 'test') ? '' : 'payu_paisa';

    $billing_profile = $order->getBillingProfile();
    $address = NULL;
    if ($billing_profile && !$billing_profile->get('address')->isEmpty()) {
      $address = $billing_profile->get('address')->first();
    }

    // Prepare the payments parameters.
    $parameters = [
      'key' => $key,
      'hash' => '',
      'txnid' => substr(hash('sha256', mt_rand() . microtime()), 0, 20),
      'amount' => round($payment->getAmount()->getNumber(), 2),
      'productinfo' => 'ProductorderID' . $order->id(),
      'firstname' => $address ? $address->getGivenName() : '',
      'lastname' => $address ? $address->getFamilyName() : '',
      'address1' => $address ? $address->getAddressLine1() : '',
      'address2' => '',
      'city' => $address ? $address->getLocality() : '',
      'state' => $address ? $address->getAdministrativeArea() : '',
      'country' => $address ? $address->getCountryCode() : '',
      'zipcode' => $address ? $address->getPostalCode() : '',
      'email' => $order->getEmail(),
      'phone' => '',
      'udf1' => '',
      'udf2' => '',
      'udf3' => '',
      'udf4' => '',
      'udf5' => '',
      'udf6' => '',
      'udf7' => '',
      'udf8' => '',
      'udf9' => '',
      'udf10' => '',
      'surl' => Url::fromRoute('commerce_payment.checkout.return', [
        'commerce_order' => $order->id(),
        'step' => 'payment'
      ], ['absolute' => TRUE])->toString(),
      'furl' => Url::fromRoute('commerce_payment.checkout.cancel', [
        'commerce_order' => $order->id(),
        'step' => 'payment'
      ], ['absolute' => TRUE])->toString(),
      'service_provider' => $service_provider,
    ];

    // Hash Sequence.
    $hashSequence = "key|txnid|amount|productinfo|firstname|email|udf1|udf2|udf3|udf4|udf5|udf6|udf7|udf8|udf9|udf10";
    $hashVarsSeq = explode('|', $hashSequence);
    $hash_string = '';
    foreach ($hashVarsSeq as $hash_var) {
      $hash_string .= isset($parameters[$hash_var]) ? $parameters[$hash_var] : '';
      $hash_string .= '|';
    }
    $hash_string .= $salt;
    $hash = strtolower(hash('sha512', $hash_string));

    $parameters['hash'] = !empty($hash) ? $hash : '';

    $redirect_url = ($mode == 'test') ? self::PAYUMONEY_API_TEST_URL : self::PAYUMONEY_API_URL;

    // Build a redirect form that will POST to the payment gateway.
    $form['#action'] = $redirect_url;
    $form['#method'] = $redirect_method;

    foreach ($parameters as $key => $value) {
      $form[$key] = [
        '#type' => 'hidden',
        '#value' => $value,
      ];
    }

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Proceed to PayUMoney'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // This form redirects to PayUMoney, so no submit handling needed here
    // The actual processing happens on the return URL
  }

  /**
   * Sets the payment entity.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentInterface $payment
   *   The payment entity.
   */
  public function setPayment($payment) {
    $this->payment = $payment;
  }

}
