<?php

namespace Drupal\Tests\commerce_payumoney\Kernel;

use Drupal\Tests\commerce\Kernel\CommerceKernelTestBase;
use Drupal\commerce_payment\Entity\PaymentGateway;

/**
 * Tests the PayUMoney payment gateway.
 *
 * @group commerce_payumoney
 */
class PayUMoneyRedirectTest extends CommerceKernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'commerce_payumoney',
    'commerce_payment',
    'commerce_order',
  ];

  /**
   * The payment gateway.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $gateway;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('commerce_payment_gateway');
    $this->installConfig(['commerce_payumoney']);

    // Create a PayUMoney payment gateway.
    $this->gateway = PaymentGateway::create([
      'id' => 'payumoney_test',
      'label' => 'PayUMoney Test',
      'plugin' => 'payumoney_redirect',
      'configuration' => [
        'pkey' => 'test_key',
        'psalt' => 'test_salt',
        'mode' => 'test',
        'service_provider' => 'payu_paisa',
      ],
      'conditions' => [],
    ]);
    $this->gateway->save();
  }

  /**
   * Tests the payment gateway plugin configuration.
   */
  public function testGatewayConfiguration() {
    /** @var \Drupal\commerce_payumoney\Plugin\Commerce\PaymentGateway\PayUMoneyRedirect $plugin */
    $plugin = $this->gateway->getPlugin();

    $configuration = $plugin->getConfiguration();
    $this->assertEquals('test_key', $configuration['pkey']);
    $this->assertEquals('test_salt', $configuration['psalt']);
    $this->assertEquals('payu_paisa', $configuration['service_provider']);
  }

  /**
   * Tests the default configuration.
   */
  public function testDefaultConfiguration() {
    /** @var \Drupal\commerce_payumoney\Plugin\Commerce\PaymentGateway\PayUMoneyRedirect $plugin */
    $plugin = $this->gateway->getPlugin();

    $default_config = $plugin->defaultConfiguration();
    $this->assertArrayHasKey('pkey', $default_config);
    $this->assertArrayHasKey('psalt', $default_config);
    $this->assertArrayHasKey('service_provider', $default_config);
    $this->assertEquals('', $default_config['pkey']);
    $this->assertEquals('', $default_config['psalt']);
    $this->assertEquals('payu_paisa', $default_config['service_provider']);
  }

  /**
   * Tests the configuration form.
   */
  public function testConfigurationForm() {
    /** @var \Drupal\commerce_payumoney\Plugin\Commerce\PaymentGateway\PayUMoneyRedirect $plugin */
    $plugin = $this->gateway->getPlugin();

    $form = [];
    $form_state = $this->createMock('Drupal\Core\Form\FormStateInterface');

    $result = $plugin->buildConfigurationForm($form, $form_state);

    $this->assertArrayHasKey('pkey', $result);
    $this->assertArrayHasKey('psalt', $result);
    $this->assertArrayHasKey('service_provider', $result);
    $this->assertEquals('textfield', $result['pkey']['#type']);
    $this->assertEquals('textfield', $result['psalt']['#type']);
    $this->assertEquals('select', $result['service_provider']['#type']);
  }

  /**
   * Tests hash generation functionality.
   */
  public function testHashGeneration() {
    $parameters = [
      'key' => 'test_key',
      'txnid' => 'test_txn_123',
      'amount' => '100.00',
      'productinfo' => 'Test Product',
      'firstname' => 'John',
      'email' => 'john@example.com',
      'udf1' => '',
      'udf2' => '',
      'udf3' => '',
      'udf4' => '',
      'udf5' => '',
      'udf6' => '',
      'udf7' => '',
      'udf8' => '',
      'udf9' => '',
      'udf10' => '',
    ];

    $salt = 'test_salt';

    // Build hash string as PayUMoney expects
    $hash_string = implode('|', [
      $parameters['key'],
      $parameters['txnid'],
      $parameters['amount'],
      $parameters['productinfo'],
      $parameters['firstname'],
      $parameters['email'],
      $parameters['udf1'],
      $parameters['udf2'],
      $parameters['udf3'],
      $parameters['udf4'],
      $parameters['udf5'],
      $parameters['udf6'],
      $parameters['udf7'],
      $parameters['udf8'],
      $parameters['udf9'],
      $parameters['udf10'],
      $salt
    ]);

    $expected_hash = strtolower(hash('sha512', $hash_string));

    // Verify hash properties
    $this->assertIsString($expected_hash);
    $this->assertEquals(128, strlen($expected_hash)); // SHA512 produces 128-character hex string
    $this->assertMatchesRegularExpression('/^[a-f0-9]{128}$/', $expected_hash);
  }

}
