<?php

namespace Drupal\commerce_pesapal\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\commerce_payment\Entity\PaymentGatewayInterface;
use Drupal\commerce_pesapal\Plugin\Commerce\PaymentGateway\PesapalRedirect;

/**
 * Simple UI form to test Pesapal QueryPaymentStatus.
 */
class PesapalStatusTestForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'commerce_pesapal_status_test_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $options = [];
    $storage = \Drupal::entityTypeManager()->getStorage('commerce_payment_gateway');
    /** @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface[] $gateways */
    $gateways = $storage->loadMultiple();
    foreach ($gateways as $gateway) {
      $plugin = $gateway->getPlugin();
      if ($plugin instanceof PesapalRedirect) {
        $options[$gateway->id()] = $gateway->label() . ' (' . $gateway->id() . ')';
      }
    }

    if (empty($options)) {
      $form['message'] = [
        '#markup' => $this->t('No Pesapal payment gateways are configured. Please create one first.'),
      ];
      return $form;
    }

    $form['gateway_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Payment gateway'),
      '#options' => $options,
      '#required' => TRUE,
      '#default_value' => array_key_first($options),
    ];

    $form['merchant_reference'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant reference'),
      '#description' => $this->t('Usually the Commerce order ID that Pesapal will send back.'),
      '#required' => TRUE,
    ];

    $form['tracking_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Transaction tracking ID'),
      '#description' => $this->t('Pesapal transaction tracking ID (pesapal_transaction_tracking_id).'),
      '#required' => TRUE,
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Test status'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $gateway_id = $form_state->getValue('gateway_id');
    $merchant_reference = trim($form_state->getValue('merchant_reference'));
    $tracking_id = trim($form_state->getValue('tracking_id'));

    /** @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface $gateway */
    $gateway = \Drupal::entityTypeManager()
      ->getStorage('commerce_payment_gateway')
      ->load($gateway_id);

    if (!$gateway) {
      $this->messenger()->addError($this->t('Selected payment gateway could not be loaded.'));
      return;
    }

    $plugin = $gateway->getPlugin();
    if (!$plugin instanceof PesapalRedirect) {
      $this->messenger()->addError($this->t('Selected gateway is not a Pesapal redirect gateway.'));
      return;
    }

    $status = $plugin->testQueryPaymentStatus($merchant_reference, $tracking_id);

    if ($status === NULL) {
      $this->messenger()->addError($this->t('No status could be retrieved. Check logs for details.'));
    }
    else {
      $this->messenger()->addStatus($this->t('Pesapal status for reference @ref / tracking @tracking: @status', [
        '@ref' => $merchant_reference,
        '@tracking' => $tracking_id,
        '@status' => $status,
      ]));
    }
  }

}
