<?php

namespace Drupal\commerce_pesapal\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsNotificationsInterface;
use Drupal\commerce_price\Price;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Provides the Pesapal off-site redirect payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "pesapal_redirect",
 *   label = @Translation("Pesapal (Redirect)"),
 *   display_label = @Translation("Pesapal"),
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_pesapal\PluginForm\OffsiteRedirect\PesapalRedirectForm"
 *   },
 *   modes = {
 *     "test" = @Translation("Sandbox"),
 *     "live" = @Translation("Live")
 *   }
 * )
 */
class PesapalRedirect extends OffsitePaymentGatewayBase implements SupportsNotificationsInterface {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'consumer_key_test' => '',
      'consumer_secret_test' => '',
      'consumer_key_live' => '',
      'consumer_secret_live' => '',
      'ipn_url' => '',
      'logging_verbosity' => 'all',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    // Current plugin configuration.
    $configuration = $this->getConfiguration();

    // SANDBOX CREDS GROUP (visual only).
    $form['sandbox_credentials'] = [
      '#type' => 'details',
      '#title' => $this->t('Sandbox credentials'),
      '#open' => TRUE,
    ];
    $form['sandbox_credentials']['consumer_key_test'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sandbox consumer key'),
      '#description' => $this->t('Pesapal API consumer key for the Sandbox (demo) environment.'),
      '#default_value' => $configuration['consumer_key_test'] ?? '',
    ];
    $form['sandbox_credentials']['consumer_secret_test'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sandbox consumer secret'),
      '#description' => $this->t('Pesapal API consumer secret for the Sandbox (demo) environment.'),
      '#default_value' => $configuration['consumer_secret_test'] ?? '',
    ];

    // LIVE CREDS GROUP (visual only).
    $form['live_credentials'] = [
      '#type' => 'details',
      '#title' => $this->t('Live credentials'),
      '#open' => FALSE,
    ];
    $form['live_credentials']['consumer_key_live'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Live consumer key'),
      '#description' => $this->t('Pesapal API consumer key for the Live (production) environment.'),
      '#default_value' => $configuration['consumer_key_live'] ?? '',
    ];
    $form['live_credentials']['consumer_secret_live'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Live consumer secret'),
      '#description' => $this->t('Pesapal API consumer secret for the Live (production) environment.'),
      '#default_value' => $configuration['consumer_secret_live'] ?? '',
    ];

    // Mode help (purely informational).
    $form['mode_help'] = [
      '#type' => 'item',
      '#title' => $this->t('Environment'),
      '#markup' => $this->t('Use the "Mode" selector above to switch between Sandbox (demo.pesapal.com) and Live (www.pesapal.com). The corresponding credentials will be used automatically.'),
    ];

    // IPN URL override.
    $form['ipn_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('IPN URL override'),
      '#description' => $this->t('Optional override for the IPN URL sent to Pesapal. Leave empty to use the default route /payment/pesapal/ipn.'),
      '#default_value' => $configuration['ipn_url'] ?? '',
    ];

    // IPN help details (no config stored).
    $form['ipn_help'] = [
      '#type' => 'details',
      '#title' => $this->t('How to configure IPN in the Pesapal dashboard'),
      '#open' => FALSE,
      '#description' => $this->t('In your Pesapal merchant dashboard, configure the Instant Payment Notification (IPN) or callback URL as follows:<br/>
        <ol>
          <li>Log into your Pesapal merchant account.</li>
          <li>Go to the settings / IPN / notifications section (naming may vary by account type).</li>
          <li>Set the notification or IPN URL to the route on this site: <code>@default_ipn</code>.</li>
          <li>If you set a custom &quot;IPN URL override&quot; above, use that exact URL instead.</li>
          <li>Save the changes and run a test transaction from Sandbox to confirm that IPN hits this site (check the <code>commerce_pesapal</code> log channel).</li>
        </ol>', [
        '@default_ipn' => \Drupal::request()->getSchemeAndHttpHost() . '/payment/pesapal/ipn',
      ]),
    ];

    // Logging verbosity.
    $form['logging_verbosity'] = [
      '#type' => 'radios',
      '#title' => $this->t('Logging verbosity'),
      '#description' => $this->t('Control how much is logged to the commerce_pesapal channel. In production you may prefer "Errors only" or "None".'),
      '#options' => [
        'all' => $this->t('All events (recommended for testing)'),
        'errors' => $this->t('Errors only'),
        'none' => $this->t('No logging'),
      ],
      '#default_value' => $configuration['logging_verbosity'] ?? 'all',
    ];

    return $form;
  }

    /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    // Let the parent pull the configuration subtree using $form['#parents'].
    parent::submitConfigurationForm($form, $form_state);

    // Values for this plugin live under the parents of this form;
    // in Commerce this is usually ['configuration'].
    $values = $form_state->getValue($form['#parents']) ?? [];

    // Our credential fields are inside the sandbox_credentials and live_credentials
    // details groups in the form, so read from those nested arrays.
    $sandbox = $values['sandbox_credentials'] ?? [];
    $live = $values['live_credentials'] ?? [];

    $this->configuration['consumer_key_test'] = $sandbox['consumer_key_test'] ?? '';
    $this->configuration['consumer_secret_test'] = $sandbox['consumer_secret_test'] ?? '';
    $this->configuration['consumer_key_live'] = $live['consumer_key_live'] ?? '';
    $this->configuration['consumer_secret_live'] = $live['consumer_secret_live'] ?? '';

    // These are direct children of the configuration form, so read them directly.
    $this->configuration['ipn_url'] = $values['ipn_url'] ?? '';
    $this->configuration['logging_verbosity'] = $values['logging_verbosity'] ?? 'all';
  }

  /**
   * Gets the Pesapal base URL from the gateway mode.
   *
   * @return string
   *   The base URL for the current mode.
   */
  public function getBaseUrl() {
    if (method_exists($this, 'getMode')) {
      $mode = $this->getMode();
    }
    else {
      $mode = $this->configuration['mode'] ?? 'test';
    }

    return $mode === 'live'
      ? 'https://www.pesapal.com'
      : 'https://demo.pesapal.com';
  }

  /**
   * Returns the consumer key/secret pair for the current mode.
   *
   * @return array
   *   An array with keys 'key', 'secret', and 'mode'.
   */
  public function getCredentialsForMode() {
    if (method_exists($this, 'getMode')) {
      $mode = $this->getMode();
    }
    else {
      $mode = $this->configuration['mode'] ?? 'test';
    }

    if ($mode === 'live') {
      return [
        'key' => $this->configuration['consumer_key_live'],
        'secret' => $this->configuration['consumer_secret_live'],
        'mode' => 'live',
      ];
    }

    return [
      'key' => $this->configuration['consumer_key_test'],
      'secret' => $this->configuration['consumer_secret_test'],
      'mode' => 'test',
    ];
  }

  /**
   * Logs to the commerce_pesapal channel respecting logging_verbosity.
   *
   * @param string $level
   *   The log level method name (e.g. 'notice', 'error').
   * @param string $message
   *   The message string.
   * @param array $context
   *   Context values.
   */
  public function log($level, $message, array $context = []) {
    $verbosity = $this->configuration['logging_verbosity'] ?? 'all';

    if ($verbosity === 'none') {
      return;
    }
    if ($verbosity === 'errors' && $level !== 'error') {
      return;
    }

    \Drupal::logger('commerce_pesapal')->$level($message, $context);
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    $merchant_reference = $request->query->get('pesapal_merchant_reference') ?: $order->id();
    $tracking_id = $request->query->get('pesapal_transaction_tracking_id');

    if (!empty($tracking_id)) {
      $status = $this->queryPaymentStatus((string) $merchant_reference, (string) $tracking_id);
      $this->log('notice', 'Return from Pesapal for order @order with tracking @tracking: status @status', [
        '@order' => $merchant_reference,
        '@tracking' => $tracking_id,
        '@status' => $status ?? 'UNKNOWN',
      ]);

      if ($status === 'COMPLETED') {
        $this->ensureSuccessfulPayment($order, $tracking_id);
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request) {
    $notification = $request->query->get('pesapal_notification_type');
    $tracking_id = $request->query->get('pesapal_transaction_tracking_id');
    $merchant_reference = $request->query->get('pesapal_merchant_reference');

    $this->log('notice', 'IPN from Pesapal: type=@type, reference=@ref, tracking=@tracking', [
      '@type' => $notification,
      '@ref' => $merchant_reference,
      '@tracking' => $tracking_id,
    ]);

    if ($notification === 'CHANGE' && !empty($tracking_id) && !empty($merchant_reference)) {
      $status = $this->queryPaymentStatus((string) $merchant_reference, (string) $tracking_id);

      if ($status === 'COMPLETED') {
        $order_storage = $this->entityTypeManager->getStorage('commerce_order');
        /** @var \Drupal\commerce_order\Entity\OrderInterface|null $order */
        $order = $order_storage->load($merchant_reference);
        if ($order) {
          $this->ensureSuccessfulPayment($order, $tracking_id);
        }
      }
      else {
        $this->log('notice', 'Pesapal status for order @order / tracking @tracking: @status', [
          '@order' => $merchant_reference,
          '@tracking' => $tracking_id,
          '@status' => $status ?? 'UNKNOWN',
        ]);
      }
    }

    $ack = http_build_query([
      'pesapal_notification_type' => $notification,
      'pesapal_transaction_tracking_id' => $tracking_id,
      'pesapal_merchant_reference' => $merchant_reference,
    ]);

    return new Response($ack);
  }

  /**
   * Ensures there is a completed payment for this order / transaction
   * and marks the order as Completed only when we have a Pesapal transaction ID.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   * @param string $tracking_id
   *   The Pesapal transaction tracking ID.
   */
  protected function ensureSuccessfulPayment(OrderInterface $order, $tracking_id) {
    // Extra safety: never do anything without a transaction ID from Pesapal.
    if (empty($tracking_id)) {
      $this->log('error', 'Tried to complete order @order without a Pesapal transaction ID.', [
        '@order' => $order->id(),
      ]);
      return;
    }

    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');

    // Avoid duplicating payments for the same remote transaction.
    $existing = $payment_storage->loadByProperties([
      'order_id' => $order->id(),
      'remote_id' => $tracking_id,
    ]);
    if ($existing) {
      // If we already have a payment for this tracking ID, don't create another.
      return;
    }

    $total_price = $order->getTotalPrice();
    $amount = new Price($total_price->getNumber(), $total_price->getCurrencyCode());

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $payment_storage->create([
      'state' => 'completed',
      'amount' => $amount,
      'payment_gateway' => $this->parentEntity->id(),
      'order_id' => $order->id(),
      'remote_id' => $tracking_id,
      'remote_state' => 'completed',
    ]);
    $payment->save();

    // Now that we have a confirmed Pesapal transaction ID and a completed payment,
    // move the order to the "completed" state (label: Completed).
    try {
      $current_state = $order->getState()->value ?? NULL;
      if ($current_state !== 'completed') {
        $order->set('state', 'completed');
        $order->save();
      }
    }
    catch (\Exception $e) {
      // Don't break payment creation if state change fails.
      $this->log('error', 'Failed to mark order @order as Completed after successful Pesapal payment: @message', [
        '@order' => $order->id(),
        '@message' => $e->getMessage(),
      ]);
    }

    $this->log('notice', 'Created completed payment for order @order with tracking @tracking and amount @amount @currency', [
      '@order' => $order->id(),
      '@tracking' => $tracking_id,
      '@amount' => $amount->getNumber(),
      '@currency' => $amount->getCurrencyCode(),
    ]);
  }
  /**
   * Queries Pesapal for the current payment status via OAuth.
   *
   * @param string $merchant_reference
   *   The merchant reference (usually the order ID).
   * @param string $tracking_id
   *   The Pesapal transaction tracking ID.
   *
   * @return string|null
   *   The status (PENDING, COMPLETED, FAILED, INVALID) or NULL on error.
   */
  /**
   * Public wrapper used by the admin status tester UI.
   *
   * @param string $merchant_reference
   *   The merchant reference.
   * @param string $tracking_id
   *   The transaction tracking ID.
   *
   * @return string|null
   *   The status string or NULL on error.
   */
  public function testQueryPaymentStatus($merchant_reference, $tracking_id) {
    return $this->queryPaymentStatus($merchant_reference, $tracking_id);
  }

  protected function queryPaymentStatus($merchant_reference, $tracking_id) {
    $module_handler = \Drupal::service('extension.list.module');
    $module_path = $module_handler->getPath('commerce_pesapal');
    require_once DRUPAL_ROOT . '/' . $module_path . '/includes/OAuth.php';

    $creds = $this->getCredentialsForMode();
    $consumer_key = $creds['key'];
    $consumer_secret = $creds['secret'];
    $mode = $creds['mode'];

    $base_url = rtrim($this->getBaseUrl(), '/');
    $endpoint = $base_url . '/API/QueryPaymentStatus';

    $consumer = new \OAuthConsumer($consumer_key, $consumer_secret);
    $signature_method = new \OAuthSignatureMethod_HMAC_SHA1();

    $request = \OAuthRequest::from_consumer_and_token($consumer, NULL, 'GET', $endpoint);
    $request->set_parameter('pesapal_merchant_reference', $merchant_reference);
    $request->set_parameter('pesapal_transaction_tracking_id', $tracking_id);
    $request->sign_request($signature_method, $consumer, NULL);

    $url = (string) $request;

    try {
      $client = \Drupal::httpClient();
      $response = $client->get($url, ['timeout' => 30]);
      $body = (string) $response->getBody();
    }
    catch (\Exception $e) {
      $this->log('error', 'Pesapal status query failed for @ref in @mode mode: @message', [
        '@ref' => $merchant_reference,
        '@mode' => $mode,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }

    $status = NULL;
    parse_str($body, $parsed);
    if (!empty($parsed['pesapal_response_data'])) {
      $status = trim($parsed['pesapal_response_data']);
    }
    else {
      $trimmed = trim($body);
      if (in_array($trimmed, ['PENDING', 'COMPLETED', 'FAILED', 'INVALID'], TRUE)) {
        $status = $trimmed;
      }
    }

    $this->log('notice', 'Queried Pesapal status for order @order / tracking @tracking in @mode mode: @status (raw="@raw")', [
      '@order' => $merchant_reference,
      '@tracking' => $tracking_id,
      '@mode' => $mode,
      '@status' => $status ?? 'UNKNOWN',
      '@raw' => trim($body),
    ]);

    return $status;
  }

}
