<?php

namespace Drupal\commerce_pesapal\PluginForm\OffsiteRedirect;

use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Provides the Pesapal redirect form.
 *
 * Responsible for building the OAuth-signed redirect URL to Pesapal when the
 * customer clicks "Pay and complete purchase".
 */
class PesapalRedirectForm extends PaymentOffsiteForm {

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->entity;
    $order = $payment->getOrder();
    $gateway = $payment->getPaymentGateway();
    $gateway_plugin = $gateway->getPlugin();
    $configuration = $gateway_plugin->getConfiguration();

    // Load the bundled OAuth 1.0 library used by Pesapal.
    $module_handler = \Drupal::service('extension.list.module');
    $module_path = $module_handler->getPath('commerce_pesapal');
    require_once DRUPAL_ROOT . '/' . $module_path . '/includes/OAuth.php';

    // Build the transaction details.
    $amount = $order->getTotalPrice();
    $amount_number = (float) $amount->getNumber();
    $amount_formatted = number_format($amount_number, 2, '.', '');
    $currency = $amount->getCurrencyCode();
    $reference = $order->id();

    $email = $order->getEmail();
    $first_name = '';
    $last_name = '';

    $billing_profile = $order->getBillingProfile();
    if ($billing_profile && !$billing_profile->get('address')->isEmpty()) {
      $address = $billing_profile->get('address')->first();
      $first_name = $address->getGivenName();
      $last_name = $address->getFamilyName();
    }

    // Use the return URL provided by Commerce as the Pesapal callback URL.
    $return_url = $form['#return_url'];
    if ($return_url instanceof Url) {
      $return_url = $return_url->setAbsolute()->toString();
    }
    else {
      $return_url = (string) $return_url;
    }

    // Build the XML payload according to Pesapal docs.
    $description = 'Order #' . $order->id();
    $post_xml = "<?xml version=\"1.0\" encoding=\"utf-8\"?>" .
      "<PesapalDirectOrderInfo xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" " .
      "xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" " .
      "Amount=\"" . htmlspecialchars($amount_formatted, ENT_QUOTES, 'UTF-8') . "\" " .
      "Description=\"" . htmlspecialchars($description, ENT_QUOTES, 'UTF-8') . "\" " .
      "Type=\"MERCHANT\" " .
      "Reference=\"" . htmlspecialchars($reference, ENT_QUOTES, 'UTF-8') . "\" " .
      "FirstName=\"" . htmlspecialchars($first_name, ENT_QUOTES, 'UTF-8') . "\" " .
      "LastName=\"" . htmlspecialchars($last_name, ENT_QUOTES, 'UTF-8') . "\" " .
      "Email=\"" . htmlspecialchars($email, ENT_QUOTES, 'UTF-8') . "\" " .
      "PhoneNumber=\"\" " .
      "Currency=\"" . htmlspecialchars($currency, ENT_QUOTES, 'UTF-8') . "\" " .
      "xmlns=\"http://www.pesapal.com\" />";

    // Pesapal expects the XML to be HTML-entity encoded.
    $post_xml = htmlentities($post_xml);

    // Credentials depend on gateway mode: Sandbox vs Live.
    $creds = $gateway_plugin->getCredentialsForMode();
    $consumer_key = $creds['key'];
    $consumer_secret = $creds['secret'];
    $mode = $creds['mode'];

    $base_url = rtrim($gateway_plugin->getBaseUrl(), '/');
    $endpoint = $base_url . '/API/PostPesapalDirectOrderV4';

    $consumer = new \OAuthConsumer($consumer_key, $consumer_secret);
    $signature_method = new \OAuthSignatureMethod_HMAC_SHA1();

    $request = \OAuthRequest::from_consumer_and_token($consumer, NULL, 'GET', $endpoint);
    $request->set_parameter('oauth_callback', $return_url);
    $request->set_parameter('pesapal_request_data', $post_xml);
    $request->sign_request($signature_method, $consumer, NULL);

    $redirect_url = (string) $request;

    // Delegate logging respect to the gateway's logging_verbosity.
    if (method_exists($gateway_plugin, 'log')) {
      $gateway_plugin->log('notice', 'Redirecting to Pesapal for order @order in @mode mode: @url', [
        '@order' => $order->id(),
        '@mode' => $mode,
        '@url' => $redirect_url,
      ]);
    }

    return $this->buildRedirectForm(
      $form,
      $form_state,
      $redirect_url,
      [],
      self::REDIRECT_GET
    );
  }

}
