<?php

namespace Drupal\commerce_pricelist_schedule\Controller;

use Drupal\commerce_pricelist_schedule\Entity\PricelistScheduledImport;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for listing scheduled imports.
 */
class ScheduledImportListController extends ControllerBase {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The file URL generator service.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * Constructs a ScheduledImportListController object.
   */
  public function __construct(DateFormatterInterface $date_formatter, EntityTypeManagerInterface $entity_type_manager, FileUrlGeneratorInterface $file_url_generator) {
    $this->dateFormatter = $date_formatter;
    $this->entityTypeManager = $entity_type_manager;
    $this->fileUrlGenerator = $file_url_generator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('date.formatter'),
      $container->get('entity_type.manager'),
      $container->get('file_url_generator')
    );
  }

  /**
   * Displays the scheduled imports for a given pricelist.
   */
  public function list($commerce_pricelist) {
    $build = [];

    $header = [
      $this->t('ID'),
      $this->t('Status'),
      $this->t('Scheduled Time'),
      $this->t('CSV'),
      $this->t('Delete existing?'),
      $this->t('Operations'),
    ];

    $rows = [];

    $query = $this->entityTypeManager->getStorage('pricelist_scheduled_import')
      ->getQuery();
    $ids = $query
      ->accessCheck(TRUE)
      ->condition('commerce_pricelist', $commerce_pricelist->id())
      ->sort('scheduled_time', 'DESC')
      ->execute();

    $imports = $this->entityTypeManager->getStorage('pricelist_scheduled_import')
      ->loadMultiple($ids);
    foreach ($imports as $import) {
      /** @var \Drupal\commerce_pricelist_schedule\Entity\PricelistScheduledImport $import */
      $operations = [];
      if ($import->get('status')->value == PricelistScheduledImport::STATUS_PENDING) {
        $url = Url::fromRoute('commerce_pricelist_schedule.cancel_import', [
          'commerce_pricelist' => $commerce_pricelist->id(),
          'pricelist_scheduled_import' => $import->id(),
        ]);
        $operations[] = Link::fromTextAndUrl($this->t('Cancel'), $url)
          ->toString();
      }

      // Generate a link to the CSV file if available.
      $csv_link = $this->t('N/A');
      if (!$import->get('import_file')->isEmpty()) {
        $file = $import->get('import_file')->entity;
        if ($file) {
          $file_url = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
          $csv_link = Link::fromTextAndUrl($this->t('Download CSV'), Url::fromUri($file_url))->toString();
        }
      }
      $import_settings = $import->get('import_settings')->value ? json_decode($import->get('import_settings')->value, TRUE) : [];
      $rows[] = [
        $import->id(),
        $import->get('status')->value,
        $this->dateFormatter->format(strtotime($import->get('scheduled_time')->value), 'short'),
        $csv_link,
        !empty($import_settings['delete_fieldset']['delete_existing']) ? $this->t('Yes') : $this->t('No'),
        ['data' => ['#markup' => implode(' ', $operations)]],
      ];
    }

    $build['imports_table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No imports are scheduled.'),
    ];

    $build['old_imports_info'] = [
      '#markup' => $this->t('Note: completed, failed, and cancelled imports older than one month are automatically removed.'),
    ];

    return $build;
  }

}
