<?php

namespace Drupal\commerce_pricelist_schedule\Form;

use Drupal\commerce_pricelist\Entity\PriceListInterface;
use Drupal\commerce_pricelist\Form\PriceListItemImportForm;
use Drupal\commerce_pricelist_schedule\Entity\PricelistScheduledImport;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StreamWrapper\PrivateStream;

/**
 * Form for scheduling a pricelist import.
 */
class ScheduleImportForm extends PriceListItemImportForm {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'commerce_pricelist_schedule_import_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, PriceListInterface $commerce_pricelist = NULL) {

    $form = parent::buildForm($form, $form_state, $commerce_pricelist);

    $form['csv']['#upload_location'] = PrivateStream::basePath() ? 'private://pricelists/' : 'temporary://pricelists/';

    // Scheduled start time field.
    $form['scheduled_time'] = [
      '#type' => 'datetime',
      '#title' => $this->t('Scheduled Start Time'),
      '#description' => $this->t('Select the date and time when the import should start.'),
      '#required' => TRUE,
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Schedule Import'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    \Drupal::service('file_system')->prepareDirectory($form['csv']['#upload_location'], FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $file_storage = $this->entityTypeManager->getStorage('file');
    /** @var \Drupal\file\FileInterface $file */
    $file = $file_storage->load($form_state->get('csv_file'));
    if ($file) {
      $file->setPermanent();
      $file->save();
    }

    $import_settings = $form_state->getValues();
    unset($import_settings['csv'], $import_settings['actions'], $import_settings['form_build_id'], $import_settings['form_token'], $import_settings['form_id'], $import_settings['op']);

    $commerce_pricelist_id = $form_state->get('price_list_id');

    $scheduled_time = $form_state->getValue('scheduled_time');

    $scheduled_import = PricelistScheduledImport::create([
      'commerce_pricelist' => $commerce_pricelist_id,
      'import_file' => $file,
      'import_settings' => json_encode($import_settings),
      'scheduled_time' => $scheduled_time->format('Y-m-d\TH:i:s'),
      'status' => PricelistScheduledImport::STATUS_PENDING,
    ]);
    $scheduled_import->save();

    // Record file usage to prevent garbage collection.
    if (!empty($file)) {
      \Drupal::service('file.usage')
        ->add($file, 'commerce_pricelist_schedule', 'pricelist_scheduled_import', $scheduled_import->id());
    }

    $this->messenger()->addMessage($this->t('Import scheduled successfully.'));

    $form_state->setRedirect('commerce_pricelist_schedule.scheduled_import_list', ['commerce_pricelist' => $commerce_pricelist_id]);
  }

}
