<?php

namespace Drupal\commerce_pricelist_schedule\Form;

use Drupal\commerce_pricelist\Entity\PriceList;
use Drupal\commerce_pricelist_schedule\Entity\PricelistScheduledImport;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Provides a confirmation form for cancelling a scheduled import.
 */
class ScheduledImportCancelConfirmForm extends ConfirmFormBase {

  /**
   * The scheduled import entity.
   *
   * @var \Drupal\commerce_pricelist_schedule\Entity\PricelistScheduledImport
   */
  protected $scheduledImport;

  /**
   * The price list entity.
   *
   * @var \Drupal\Core\Entity\EntityInterface
   */
  protected $priceList;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'commerce_pricelist_schedule_cancel_confirm_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to cancel this scheduled import?');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelText() {
    return $this->t('Back');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('commerce_pricelist_schedule.scheduled_import_list', ['commerce_pricelist' => $this->priceList->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('You will have to schedule a new import if you need to.');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Yes, cancel it');
  }

  /**
   * Builds the confirmation form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param \Drupal\Core\Entity\EntityInterface $price_list
   *   The price list entity from the route.
   * @param \Drupal\commerce_pricelist_schedule\Entity\PricelistScheduledImport $pricelist_scheduled_import
   *   The scheduled import entity from the route.
   *
   * @return array
   *   The renderable form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state, PriceList $commerce_pricelist = NULL, PricelistScheduledImport $pricelist_scheduled_import = NULL) {
    $this->priceList = $commerce_pricelist;
    $this->scheduledImport = $pricelist_scheduled_import;
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if ($this->scheduledImport->get('status')->value == PricelistScheduledImport::STATUS_PENDING) {
      $this->scheduledImport->set('status', PricelistScheduledImport::STATUS_CANCELLED);
      $this->scheduledImport->save();
      $this->messenger()->addMessage($this->t('Scheduled import has been cancelled.'));
    }
    else {
      $this->messenger()->addMessage($this->t('Only pending imports can be cancelled.'), 'error');
    }
    $form_state->setRedirect('commerce_pricelist_schedule.scheduled_import_list', ['commerce_pricelist' => $this->priceList->id()]);
  }

}
