<?php

namespace Drupal\commerce_pricelist_schedule\Plugin\QueueWorker;

use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\commerce_pricelist_schedule\Entity\PricelistScheduledImport;
use Drupal\file\Entity\File;
use Drupal\commerce_pricelist\Form\PriceListItemImportForm;

/**
 * Processes scheduled imports via a simulated batch API using PriceListItemImportForm methods.
 *
 * @QueueWorker(
 *   id = "scheduled_import_worker",
 *   title = @Translation("Scheduled Import Worker"),
 *   cron = {"time" = 60}
 * )
 */
class ScheduledImportWorker extends QueueWorkerBase {

  /**
   * Processes one batch step for a scheduled import.
   *
   * The worker mimics the Batch API by initializing a batch context (stored in state)
   * if needed, then calling the current operation callback (using methods from
   * PriceListItemImportForm) with a context variable. Depending on the value of
   * $context['finished'], it requeues the same item to continue processing or moves
   * on to the next operation. When all operations are complete, it calls finishBatch()
   * using the signature finishBatch($success, array $results, array $operations) and
   * updates the scheduled import status accordingly.
   *
   * @param array $data
   *   Item data.
   */
  public function processItem($data) {
    $id = $data['id'];
    $storage = \Drupal::entityTypeManager()->getStorage('pricelist_scheduled_import');
    /** @var \Drupal\commerce_pricelist_schedule\Entity\PricelistScheduledImport $scheduled_import */
    $scheduled_import = $storage->load($id);
    if (!$scheduled_import) {
      return;
    }

    // Use Drupal's state service to store and retrieve the batch context.
    $state = \Drupal::state();
    $context_key = 'commerce_pricelist_schedule.batch_context.' . $id;
    $context = $state->get($context_key, NULL);

    // Use the methods provided by the PriceListItemImportForm.
    $import_form_class = PriceListItemImportForm::class;

    if ($context === NULL) {
      // Initialize the batch operations based on the import settings.
      $import_settings = $scheduled_import->get('import_settings')->value;
      $settings = json_decode($import_settings, TRUE);
      if (!$settings) {
        return;
      }
      $operations = [];

      // Operation 1: Delete existing entries if requested.
      if (!empty($settings['delete_fieldset']['delete_existing'])) {
        $operations[] = [
          'callback' => [$import_form_class, 'batchDeleteExisting'],
          'args' => [$scheduled_import->get('commerce_pricelist')->target_id],
        ];
      }
      // Load the file entity.
      $file = File::load($scheduled_import->get('import_file')->target_id);
      if ($file) {
        // Operation 2: Process the import.
        $operations[] = [
          'callback' => [$import_form_class, 'batchProcess'],
          'args' => [
            $file->getFileUri(),
            $settings['mapping'],
            $settings['options'],
            $scheduled_import->get('commerce_pricelist')->target_id,
            (bool) $settings['delete_fieldset']['delete_existing'],
          ],
        ];
      }

      // Initialize the batch context.
      $context = [
        'operations' => $operations,
        'current_operation' => 0,
        'finished' => 0,
        'results' => [],
      ];
    }

    $current_index = $context['current_operation'];

    // If there are no more operations, finish the batch.
    if (!isset($context['operations'][$current_index])) {
      try {
        // Call finishBatch with the new signature.
        $import_form_class::finishBatch(TRUE, $context['results'], $context['operations']);
        $scheduled_import->set('status', PricelistScheduledImport::STATUS_COMPLETED);
        $scheduled_import->save();
      }
      catch (\Exception $e) {
        $scheduled_import->set('status', PricelistScheduledImport::STATUS_FAILED);
        $scheduled_import->save();
        \Drupal::logger('commerce_pricelist_schedule')
          ->error('Batch finish callback failed for Scheduled Import ID @id: @message', [
            '@id' => $scheduled_import->id(),
            '@message' => $e->getMessage(),
          ]);
      }
      $state->delete($context_key);
      return;
    }

    $operation = $context['operations'][$current_index];
    $callback = $operation['callback'];
    $args = $operation['args'];

    try {
      // Call the current batch callback once, passing the context by reference.
      call_user_func_array($callback, array_merge($args, [&$context]));
    }
    catch (\Exception $e) {
      $scheduled_import->set('status', PricelistScheduledImport::STATUS_FAILED);
      $scheduled_import->save();
      \Drupal::logger('commerce_pricelist_schedule')
        ->error('Batch operation failed for Scheduled Import ID @id: @message', [
          '@id' => $scheduled_import->id(),
          '@message' => $e->getMessage(),
        ]);
      $state->delete($context_key);
      return;
    }

    if ($context['finished'] < 1) {
      // The current operation is still in progress; save the context and requeue.
      $state->set($context_key, $context);
      \Drupal::queue('scheduled_import_worker')->createItem(['id' => $id]);
    }
    elseif (!empty($context['results']['error'])) {
      // The current operation has failed; set the import status to failed.
      \Drupal::logger('commerce_pricelist_schedule')
        ->error('Failed pricelist scheduled import @id: @message', [
          '@id' => $scheduled_import->id(),
          '@message' => $context['results']['error'],
        ]);

      $scheduled_import->set('status', PricelistScheduledImport::STATUS_FAILED);
      $scheduled_import->save();
      $state->delete($context_key);
    }
    else {
      // The current operation has finished; move to the next operation.
      $context['current_operation']++;

      // Invoke any hooks for the completion of an operation.
      \Drupal::moduleHandler()->invokeAll('commerce_pricelist_schedule_finished', [$context, $scheduled_import]);

      // Reset progress and results for the next operation.
      $context['finished'] = 0;
      $context['results'] = [];
      $state->set($context_key, $context);
      if (isset($context['operations'][$context['current_operation']])) {
        \Drupal::queue('scheduled_import_worker')->createItem(['id' => $id]);
      }
      else {
        // No further operations; finish the batch.
        try {
          $import_form_class::finishBatch(TRUE, $context['results'], $context['operations']);
          $scheduled_import->set('status', PricelistScheduledImport::STATUS_COMPLETED);
          $scheduled_import->save();
        }
        catch (\Exception $e) {
          $scheduled_import->set('status', PricelistScheduledImport::STATUS_FAILED);
          $scheduled_import->save();
          \Drupal::logger('commerce_pricelist_schedule')
            ->error('Failed finish callback for pricelist scheduled import @id: @message', [
              '@id' => $scheduled_import->id(),
              '@message' => $e->getMessage(),
            ]);
        }
        $state->delete($context_key);
      }
    }
  }

}
