<?php

namespace Drupal\commerce_product_review\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Implements the 'commerce_product_review_overall_rating_default' formatter.
 *
 * @FieldFormatter(
 *   id = "commerce_product_review_overall_rating_default",
 *   label = @Translation("Default"),
 *   field_types = {
 *     "commerce_product_review_overall_rating"
 *   }
 * )
 */
class OverallRatingDefaultFormatter extends FormatterBase implements ContainerFactoryPluginInterface, OverallRatingEmptyTextFormatterInterface {

  /**
   * The number formatter.
   *
   * @var \CommerceGuys\Intl\Formatter\NumberFormatterInterface
   */
  protected $numberFormatter;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->numberFormatter = $container->get('commerce_price.number_formatter');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'strip_trailing_zeroes' => FALSE,
      'empty_text' => t('Write the first review'),
      'display_link' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = [];

    $form['strip_trailing_zeroes'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Strip trailing zeroes after the decimal point.'),
      '#default_value' => $this->getSetting('strip_trailing_zeroes'),
    ];

    $form['empty_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Empty text'),
      '#description' => $this->t('Text displayed, if no published review exists for the given product.'),
      '#default_value' => $this->getSetting('empty_text'),
    ];

    $form['display_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display rating as a link to reviews page.'),
      '#default_value' => $this->getSetting('display_link'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    if ($this->getSetting('strip_trailing_zeroes')) {
      $summary[] = $this->t('Strip trailing zeroes after the decimal point.');
    }
    else {
      $summary[] = $this->t('Do not strip trailing zeroes after the decimal point.');
    }

    if ($empty_text = $this->getSetting('empty_text')) {
      $summary[] = $this->t('Empty text: @empty_text', ['@empty_text' => $empty_text]);
    }

    if ($this->getSetting('display_link')) {
      $summary[] = $this->t('Display rating as a link to reviews page.');
    }
    else {
      $summary[] = $this->t('Do not display rating as a link to reviews page.');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    $format_options = [
      'maximum_fraction_digits' => $this->getSetting('strip_trailing_zeroes') ? 0 : 1,
    ];

    /** @var \Drupal\commerce_product\Entity\ProductInterface $product */
    $product = $items->getEntity();

    /** @var \Drupal\commerce_product_review\Plugin\Field\FieldType\OverallRatingItem $item */
    foreach ($items as $delta => $item) {
      $rating = $this->t('@rating (@count)', [
        '@rating' => $this->numberFormatter->format($item->score, $format_options),
        '@count' => $item->count,
      ]);
      if ($this->getSetting('display_link')) {
        $rating_element = [
          '#type' => 'link',
          '#title' => $rating,
          '#url' => Url::fromRoute('entity.commerce_product.reviews', ['commerce_product' => $product->id()]),
        ];
      }
      else {
        $rating_element = [
          '#type' => 'markup',
          '#markup' => $rating,
        ];
      }

      // Wrap the rating element in a container to assist in theming.
      $elements[$delta]['overall_rating'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['overall-rating']],
        'rating' => $rating_element,
        '#cache' => [
          'contexts' => [
            'languages:' . LanguageInterface::TYPE_INTERFACE,
            'country',
          ],
        ],
      ];
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function getEmptyText() {
    return $this->getSetting('empty_text');
  }

}
