<?php

namespace Drupal\commerce_product_review;

use Drupal\Component\Utility\Unicode;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Link;
use Drupal\Core\Routing\RouteMatchInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a class to build a listing of product reviews for users.
 */
class ProductReviewListBuilder extends EntityListBuilder {

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * {@inheritdoc}
   */
  protected $limit = 12;

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    $instance = parent::createInstance($container, $entity_type);
    $instance->routeMatch = $container->get('current_route_match');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEntityIds() {
    $query = $this->getStorage()->getQuery()
      ->accessCheck(TRUE)
      ->sort('created', 'DESC');

    // Filter by user if the user parameter is present in the route.
    if ($user = $this->routeMatch->getParameter('user')) {
      $query->condition('uid', $user->id());
    }

    // Only add the pager if a limit is specified.
    if ($this->limit) {
      $query->pager($this->limit);
    }

    return $query->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['product'] = $this->t('Product');
    $header['review'] = $this->t('Review');
    if (!$this->routeMatch->getParameter('user')) {
      $header['status'] = $this->t('Status');
    }

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\commerce_product_review\Entity\ProductReviewInterface $entity */
    /** @var \Drupal\commerce_product\Entity\ProductInterface $product */
    if ($product = $entity->getProduct()) {
      $row['product']['data'] = Link::fromTextAndUrl($product->label(), $product->toUrl());
    }
    else {
      $row['product'] = '';
    }
    $row['review']['data'] = [
      'title' => [
        '#type' => 'html_tag',
        '#tag' => 'h5',
        '#value' => $entity->label(),
        '#attributes' => [
          'class' => ['review-title'],
        ],
      ],
      'rating' => [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#attributes' => [
          'class' => ['rateit'],
          'data-rateit-value' => $entity->getRatingValue(),
          // cspell:disable-next-line
          'data-rateit-ispreset' => 'true',
          'data-rateit-readonly' => 'true',
        ],
      ],
      'description' => [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#value' => Unicode::truncate(strip_tags($entity->getDescription()), 256, TRUE, TRUE),
        '#attributes' => [
          'class' => ['review-description'],
        ],
      ],
    ];
    if (!$this->routeMatch->getParameter('user')) {
      $row['status'] = $entity->isPublished() ? $this->t('Published') : $this->t('Unpublished');
    }

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $build = parent::render();
    $build['#attached']['library'][] = 'commerce_product_review/rateit-js';
    return $build;
  }

}
