<?php

namespace Drupal\commerce_product_review\Controller;

use Drupal\commerce_product\Entity\ProductInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\user\UserInterface;

/**
 * Defines the product review controller.
 */
class ProductReviewController extends ControllerBase {

  /**
   * The currently authenticated user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The product review storage.
   *
   * @var \Drupal\commerce_product_review\ProductReviewStorageInterface
   */
  protected $reviewStorage;

  /**
   * The product review type storage.
   *
   * @var \Drupal\commerce_product_review\ProductReviewTypeStorageInterface
   */
  protected $reviewTypeStorage;

  /**
   * Constructs a new ProductReviewController object.
   *
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The currently authenticated user.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   */
  public function __construct(AccountInterface $current_user) {
    $this->currentUser = $current_user;
    $this->reviewStorage = $this->entityTypeManager()->getStorage('commerce_product_review');
    $this->reviewTypeStorage = $this->entityTypeManager()->getStorage('commerce_product_review_type');
  }

  /**
   * Renders the review form for the given product entity.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $commerce_product
   *   The product entity.
   *
   * @return array
   *   The form structure.
   */
  public function reviewForm(ProductInterface $commerce_product) {
    $review_type = $this->reviewTypeStorage->findMatchingReviewType($commerce_product);
    $review = $this->reviewStorage->create([
      'product_id' => $commerce_product->id(),
      'type' => $review_type->id(),
    ]);

    return $this->entityFormBuilder()->getForm($review, 'add');
  }

  /**
   * Renders the review edit form for the given product entity.
   *
   * Only used for the edit action button on the product review page.
   * Since the route does not include the id of the review it must be found
   * in the list of reviews for that product based on the current user.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $commerce_product
   *   The product entity.
   *
   * @return array
   *   The form structure.
   */
  public function reviewEditForm(ProductInterface $commerce_product) {
    $review = $this->reviewStorage->loadByProductAndUser($commerce_product->id(), $this->currentUser->id());

    return $this->entityFormBuilder()->getForm(reset($review), 'edit');
  }

  /**
   * Renders the review delete form for the given product entity.
   *
   * Only used for the delete action button on the product review page.
   * Since the route does not include the id of the review it must be found
   * in the list of reviews for that product based on the current user.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $commerce_product
   *   The product entity.
   *
   * @return array
   *   The form structure.
   */
  public function reviewDeleteForm(ProductInterface $commerce_product) {
    $review = $this->reviewStorage->loadByProductAndUser($commerce_product->id(), $this->currentUser->id());

    return $this->entityFormBuilder()->getForm(reset($review), 'delete');
  }

  /**
   * Redirects anonymous users to the login form.
   *
   * Once they login they will be sent to the review form.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $commerce_product
   *   The product entity.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the login page.
   */
  public function reviewInvitation(ProductInterface $commerce_product) {
    return $this->redirect('user.login', [], [
      'query' => [
        'destination' => Url::fromRoute('entity.commerce_product.review_form', [
          'commerce_product' => $commerce_product->id(),
        ])->toString(),
      ],
    ]);
  }

  /**
   * Renders the review page for the given product entity.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $commerce_product
   *   The product entity.
   *
   * @return array
   *   The rendered product review page.
   */
  public function reviewPage(ProductInterface $commerce_product) {
    $build = [];
    $reviews = $this->reviewStorage->loadByProductId($commerce_product->id());
    if (empty($reviews)) {
      $build['empty'] = [
        '#theme' => 'commerce_product_review_empty_page',
        '#product' => $commerce_product,
      ];
    }
    else {
      $build['reviews'] = $this->entityTypeManager()->getViewBuilder('commerce_product_review')->viewMultiple($reviews);
    }

    return $build;
  }

  /**
   * Title callback for ::reviewForm().
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $commerce_product
   *   The product entity.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The page title.
   */
  public function reviewFormTitle(ProductInterface $commerce_product) {
    return $this->t('Review product: @product', ['@product' => $commerce_product->label()]);
  }

  /**
   * Title callback for route entity.commerce_product.edit_review_form.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $commerce_product
   *   The product entity.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The page title.
   */
  public function reviewEditFormTitle(ProductInterface $commerce_product) {
    return $this->t('Edit review for: @product', ['@product' => $commerce_product->label()]);
  }

  /**
   * Title callback for ::reviewPage().
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $commerce_product
   *   The product entity.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The page title.
   */
  public function reviewPageTitle(ProductInterface $commerce_product) {
    return $this->t('Reviews for: @product', ['@product' => $commerce_product->label()]);
  }

  /**
   * Access callback for reviewForm().
   *
   * Enforces one review per product per user.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $commerce_product
   *   The product to review.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result object.
   */
  public function accessReviewForm(ProductInterface $commerce_product) {
    // Check if the product is published.
    if (!$commerce_product->isPublished()) {
      return AccessResult::forbidden()->addCacheableDependency($commerce_product);
    }

    // Check if there's a matching review type.
    $review_type = $this->reviewTypeStorage->findMatchingReviewType($commerce_product);
    if (empty($review_type)) {
      return AccessResult::forbidden()->addCacheableDependency($commerce_product);
    }

    // For authenticated users check if they've already submitted a review.
    if ($this->currentUser->isAuthenticated()) {
      $existing_reviews = $this->reviewStorage->loadByProductAndUser($commerce_product->id(), $this->currentUser->id());
      if (!empty($existing_reviews)) {
        return AccessResult::forbidden()->addCacheableDependency($commerce_product)->addCacheContexts(['user']);
      }
    }

    // If all other checks pass, check user's permission to create reviews.
    return $this->entityTypeManager()->getAccessControlHandler('commerce_product_review')->createAccess($review_type ? $review_type->id() : NULL, NULL, [], TRUE);
  }

  /**
   * Access callback for route entity.commerce_product.edit_review_form.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $commerce_product
   *   The product to review.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result object.
   */
  public function accessReviewEditForm(ProductInterface $commerce_product) {
    // Check if the product is published.
    if (!$commerce_product->isPublished()) {
      return AccessResult::forbidden()->addCacheableDependency($commerce_product);
    }

    // Deny access to anonymous users.
    if ($this->currentUser->isAnonymous()) {
      return AccessResult::forbidden()->addCacheableDependency($commerce_product)->addCacheContexts(['user.roles']);
    }

    // Check if there's a matching review type.
    $review_type = $this->reviewTypeStorage->findMatchingReviewType($commerce_product);
    if (empty($review_type)) {
      return AccessResult::forbidden()->addCacheableDependency($commerce_product);
    }

    // For authenticated users ensure they've already submitted a review.
    if ($this->currentUser->isAuthenticated()) {
      $existing_reviews = $this->reviewStorage->loadByProductAndUser($commerce_product->id(), $this->currentUser->id());
      if (!empty($existing_reviews)) {
        // Check user's permission to update their own review.
        return $this->entityTypeManager()->getAccessControlHandler('commerce_product_review')->access(reset($existing_reviews), 'update', NULL, TRUE);
      }
    }

    return AccessResult::forbidden();
  }

  /**
   * Access callback for route entity.commerce_product.delete_review_form.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $commerce_product
   *   The product to review.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result object.
   */
  public function accessReviewDeleteForm(ProductInterface $commerce_product) {
    // Check if the product is published.
    if (!$commerce_product->isPublished()) {
      return AccessResult::forbidden()->addCacheableDependency($commerce_product);
    }

    // Deny access to anonymous users.
    if ($this->currentUser->isAnonymous()) {
      return AccessResult::forbidden()->addCacheableDependency($commerce_product)->addCacheContexts(['user.roles']);
    }

    // For authenticated users ensure they've already submitted a review.
    if ($this->currentUser->isAuthenticated()) {
      $existing_reviews = $this->reviewStorage->loadByProductAndUser($commerce_product->id(), $this->currentUser->id());
      if (!empty($existing_reviews)) {
        // Check user's permission to update their own review.
        return $this->entityTypeManager()->getAccessControlHandler('commerce_product_review')->access(reset($existing_reviews), 'delete', NULL, TRUE);
      }
    }

    return AccessResult::forbidden();
  }

  /**
   * Checks access for the overview page.
   *
   * Grants access if the current user is allowed to view the specified
   * user's page.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user account.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The currently logged in account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function checkOverviewAccess(UserInterface $user, AccountInterface $account) {
    return $user->access('view', $account, TRUE);
  }

}
