<?php

namespace Drupal\commerce_product_review\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Implements the 'commerce_product_review_overall_rating_stars' formatter.
 *
 * This field formatter shows the product's overall rating as stars, based on
 * the rateit.js library.
 *
 * @FieldFormatter(
 *   id = "commerce_product_review_overall_rating_stars",
 *   label = @Translation("Stars"),
 *   field_types = {
 *     "commerce_product_review_overall_rating"
 *   }
 * )
 */
class OverallRatingStarsFormatter extends FormatterBase implements ContainerFactoryPluginInterface, OverallRatingEmptyTextFormatterInterface {

  /**
   * The current active user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The number formatter.
   *
   * @var \CommerceGuys\Intl\Formatter\NumberFormatterInterface
   */
  protected $numberFormatter;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->currentUser = $container->get('current_user');
    $instance->numberFormatter = $container->get('commerce_price.number_formatter');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'link_display' => 'overview',
      'show_review_form_link' => TRUE,
      'empty_text' => t('Write the first review'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form['link_display'] = [
      '#type' => 'radios',
      '#title' => $this->t('Link display'),
      '#options' => [
        'none' => $this->t('No link'),
        'overview' => $this->t('Full overview link (View all X reviews)'),
        'count' => $this->t('Minimal count link (X)'),
      ],
      '#default_value' => $this->getSetting('link_display'),
    ];

    $form['show_review_form_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Link to review form'),
      '#description' => $this->t('If selected, a link to the review form will be displayed, so that the visitor can write a product review.'),
      '#default_value' => $this->getSetting('show_review_form_link'),
    ];

    $form['empty_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Empty text'),
      '#description' => $this->t('Text displayed, if no published review exists for the given product.'),
      '#default_value' => $this->getSetting('empty_text'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $link_display = $this->getSetting('link_display');
    switch ($link_display) {
      case 'none':
        $summary[] = $this->t('No review link displayed.');
        break;

      case 'overview':
        $summary[] = $this->t('Display full overview link.');
        break;

      case 'count':
        $summary[] = $this->t('Display count link.');
        break;
    }

    if ($this->getSetting('show_review_form_link')) {
      $summary[] = $this->t('Display review form link.');
    }
    else {
      $summary[] = $this->t('Do not display review form link.');
    }

    if ($empty_text = $this->getSetting('empty_text')) {
      $summary[] = $this->t('Empty text: @empty_text', ['@empty_text' => $empty_text]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    /** @var \Drupal\commerce_product\Entity\ProductInterface $product */
    $product = $items->getEntity();
    /** @var \Drupal\commerce_product_review\ProductReviewStorageInterface $product_review_storage */
    $product_review_storage = $this->entityTypeManager->getStorage('commerce_product_review');
    $existing_reviews = $product_review_storage->loadByProductId($product->id(), TRUE);

    $format_options = [
      'maximum_fraction_digits' => $this->getSetting('strip_trailing_zeroes') ? 1 : 3,
    ];

    /** @var \Drupal\commerce_product_review\Plugin\Field\FieldType\OverallRatingItem $item */
    foreach ($items as $delta => $item) {
      // Define the rateit element.
      $rating = $this->numberFormatter->format($item->score, $format_options);
      $rateit_element = [
        '#type' => 'container',
        '#attributes' => ['class' => ['overall-rating-stars']],
        'overall_rating_stars' => [
          '#type' => 'html_tag',
          '#tag' => 'div',
          '#attributes' => [
            'class' => ['rateit'],
            'title' => $this->t('Overall rating: @rating', ['@rating' => $rating]),
            'data-rateit-value' => $item->score,
            // cspell:disable-next-line
            'data-rateit-ispreset' => 'true',
            'data-rateit-readonly' => 'true',
          ],
        ],
      ];

      // Generate the rating element based on the chosen display setting.
      $link_display = $this->getSetting('link_display');
      $link_url = Url::fromRoute('entity.commerce_product.reviews', ['commerce_product' => $product->id()]);
      switch ($link_display) {
        case 'overview':
          // Render rating and link separately.
          $rating_element = [
            'stars' => $rateit_element,
            'link' => [
              '#type' => 'link',
              '#attributes' => ['class' => ['overall-rating-link']],
              '#title' => $this->formatPlural(
                count($existing_reviews),
                'View @count review',
                'View all @count reviews'
              ),
              '#url' => $link_url,
            ],
          ];
          break;

        case 'count':
          // Render rating and count together in one link.
          $rateit_element['#type'] = 'html_tag';
          $rateit_element['#tag'] = 'span';
          $rating_element = [
            '#type' => 'link',
            '#title' => [
              'stars' => $rateit_element,
              'text' => [
                '#type' => 'html_tag',
                '#tag' => 'span',
                '#attributes' => ['class' => ['overall-rating-count']],
                '#value' => $this->t('(@count)', ['@count' => count($existing_reviews)]),
              ],
            ],
            '#url' => $link_url,
          ];
          break;

        default:
          // No link, just show rating.
          $rating_element = $rateit_element;
          break;
      }

      // Wrap the rating element in a container to assist in theming.
      $elements[$delta]['overall_rating'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['overall-rating']],
        'rating' => $rating_element,
      ];

      if ($this->getSetting('show_review_form_link')) {
        $existing_own_reviews = $this->currentUser->isAuthenticated() ? $product_review_storage->loadByProductAndUser($product->id(), $this->currentUser->id()) : [];
        if (empty($existing_own_reviews)) {
          $url = Url::fromRoute('entity.commerce_product.review_form', ['commerce_product' => $product->id()]);
          if ($this->currentUser->isAnonymous() && !$this->currentUser->hasPermission('create default commerce_product_review')) {
            $url = Url::fromRoute('user.login', [], ['query' => ['destination' => $url->toString()]]);
          }

          $elements[$delta]['review_form_container'] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['review-form-link']],
          ];
          $elements[$delta]['review_form_container']['review_form_link'] = [
            '#type' => 'link',
            '#title' => $this->t('Write your own review'),
            '#url' => $url,
          ];
        }
      }
    }
    $elements['#attached']['library'] = ['commerce_product_review/rateit-js'];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function getEmptyText() {
    return $this->getSetting('empty_text');
  }

}
