<?php

namespace Drupal\commerce_product_taxonomy_filter\Plugin\views\relationship;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormStateInterface;
use Drupal\taxonomy\VocabularyStorageInterface;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\Plugin\views\relationship\RelationshipPluginBase;
use Drupal\views\ViewExecutable;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Relationship handler to return the taxonomy terms of commerce_products.
 *
 * @ingroup views_relationship_handlers
 *
 * @ViewsRelationship("commerce_product_term_data")
 */
class CommerceProductTermData extends RelationshipPluginBase {

  /**
   * The vocabulary storage.
   *
   * @var \Drupal\taxonomy\VocabularyStorageInterface
   */
  protected $vocabularyStorage;

  /**
   * The join plugin manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected $joinPluginManager;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a new CommerceProductTermData relationship handler.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\taxonomy\VocabularyStorageInterface $vocabulary_storage
   *   The vocabulary storage.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $join_plugin_manager
   *   The Views join plugin manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    VocabularyStorageInterface $vocabulary_storage,
    PluginManagerInterface $join_plugin_manager,
    Connection $database,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->vocabularyStorage = $vocabulary_storage;
    $this->joinPluginManager = $join_plugin_manager;
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')->getStorage('taxonomy_vocabulary'),
      $container->get('plugin.manager.views.join'),
      $container->get('database'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, ?array &$options = NULL) {
    parent::init($view, $display, $options);

    // @todo Remove the legacy code.
    // Convert legacy vids option to machine name vocabularies.
    if (!empty($this->options['vids'])) {
      $vocabularies = $this->vocabularyStorage->loadMultiple();
      foreach ($this->options['vids'] as $vid) {
        if (isset($vocabularies[$vid])) {
          $machine_name = $vocabularies[$vid]->id();
          $this->options['vocabularies'][$machine_name] = $machine_name;
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['vids'] = ['default' => []];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    $vocabularies = $this->vocabularyStorage->loadMultiple();
    $options = [];
    foreach ($vocabularies as $voc) {
      $options[$voc->id()] = $voc->label();
    }

    $form['vids'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Vocabularies'),
      '#options' => $options,
      '#default_value' => $this->options['vids'],
      '#description' => $this->t('Choose which vocabularies you wish to relate. Remember that every term found will create a new record, so this relationship is best used on just one vocabulary that has only one term per commerce_product.'),
    ];
    parent::buildOptionsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state) {
    // Transform the #type = checkboxes value to a numerically indexed array,
    // Because the config schema expects a sequence, not a mapping.
    $vids = $form_state->getValue(['options', 'vids']);
    $form_state->setValue(['options', 'vids'], array_values(array_filter($vids)));
  }

  /**
   * Called to implement a relationship in a query.
   */
  public function query() {
    $this->ensureMyTable();

    $def = $this->definition;
    $def['table'] = 'taxonomy_term_field_data';

    if (!array_filter($this->options['vids'])) {
      $commerce_product_taxonomy_index = $this->query->addTable('commerce_product_taxonomy_index', $this->relationship);
      $def['left_table'] = $commerce_product_taxonomy_index;
      $def['left_field'] = 'tid';
      $def['field'] = 'tid';
      $def['type'] = empty($this->options['required']) ? 'LEFT' : 'INNER';
    }
    else {
      // If vocabularies are supplied join a subselect instead.
      $def['left_table'] = $this->tableAlias;
      $def['left_field'] = 'product_id';
      $def['field'] = 'product_id';
      $def['type'] = empty($this->options['required']) ? 'LEFT' : 'INNER';
      $def['adjusted'] = TRUE;

      $query = $this->database->select('taxonomy_term_field_data', 'td');
      $query->addJoin($def['type'], 'commerce_product_taxonomy_index', 'tn', 'tn.tid = td.tid');
      $query->condition('td.vid', array_filter($this->options['vids']), 'IN');
      if (empty($this->query->options['disable_sql_rewrite'])) {
        $query->addTag('taxonomy_term_access');
      }
      $query->fields('td');
      $query->fields('tn', ['product_id']);
      $def['table formula'] = $query;
    }

    // Use the injected join plugin manager.
    $join = $this->joinPluginManager->createInstance('standard', $def);

    // Use a short alias for this:
    $alias = $def['table'] . '_' . $this->table;

    $this->alias = $this->query->addRelationship($alias, $join, 'taxonomy_term_field_data', $this->relationship);
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    $dependencies = parent::calculateDependencies();

    foreach ($this->options['vids'] as $vocabulary_id) {
      if ($vocabulary = $this->vocabularyStorage->load($vocabulary_id)) {
        $dependencies[$vocabulary->getConfigDependencyKey()][] = $vocabulary->getConfigDependencyName();
      }
    }

    return $dependencies;
  }

}
