<?php

declare(strict_types=1);

namespace Drupal\Tests\commerce_profile_pane\Functional;

use Drupal\Tests\commerce\Functional\CommerceBrowserTestBase;
use Drupal\commerce_checkout\Entity\CheckoutFlow;
use Drupal\profile\Entity\ProfileType;

/**
 * Tests the profile checkout pane in the checkout flow.
 *
 * @group commerce_profile_pane
 */
class ProfilePaneCheckoutTest extends CommerceBrowserTestBase {

  /**
   * Store administrator.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * Customer used to checkout and access profile.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $customer;

  /**
   * The profile type used in tests.
   *
   * @var \Drupal\profile\Entity\ProfileType
   */
  protected $profileType;

  /**
   * The product.
   *
   * @var \Drupal\commerce_product\Entity\ProductInterface
   */
  protected $product;

  /**
   * Profile entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $profileStorage;

  /**
   * Product variation to add to cart.
   *
   * @var \Drupal\commerce_product\Entity\ProductVariationInterface
   */
  protected $variation;

  /**
   * Cart manager.
   *
   * @var \Drupal\commerce_cart\CartManagerInterface
   */
  protected $cartManager;

  /**
   * The modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'commerce_product',
    'commerce_order',
    'commerce_cart',
    'commerce_checkout',
    'views_ui',
    'commerce_profile_pane',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->createUser([
      'administer commerce_checkout_flow',
    ]);

    $this->profileStorage = \Drupal::service('entity_type.manager')->getStorage('profile');
    $this->cartManager = $this->container->get('commerce_cart.cart_manager');

    $this->placeBlock('commerce_cart');
    $this->placeBlock('commerce_checkout_progress');

    // Create a profile type.
    $this->profileType = ProfileType::create([
      'id' => $this->randomMachineName(),
      'label' => $this->randomMachineName(),
      'registration' => FALSE,
      'roles' => [],
      'multiple' => FALSE,
    ]);
    $this->profileType->save();

    // Add the pane to the checkout workflow as it defaults to "disabled".
    $checkout_flow = CheckoutFlow::load('default');
    $checkout_flow_configuration = $checkout_flow->get('configuration');
    $pane_config_key = 'profile_form:' . $this->profileType->id();
    $checkout_flow_configuration['panes'][$pane_config_key] = [
      'step' => 'order_information',
      'weight' => 3,
      'form_mode' => 'default',
    ];
    $checkout_flow->set('configuration', $checkout_flow_configuration);
    $checkout_flow->save();

    // Create the customer user.
    $this->customer = $this->createUser([
      "create {$this->profileType->id()} profile",
      "update own {$this->profileType->id()} profile",
    ]);

    // Add a field to the profile type and set it to show in the default
    // form mode.
    \Drupal::service('entity_type.manager')->getStorage('field_storage_config')->create([
      'entity_type' => 'profile',
      'field_name' => 'test_field',
      'type' => 'string',
    ])->save();
    \Drupal::service('entity_type.manager')->getStorage('field_config')->create([
      'field_name' => 'test_field',
      'entity_type' => 'profile',
      'bundle' => $this->profileType->id(),
      'label' => 'Profile field',
    ])->save();
    \Drupal::service('entity_display.repository')
      ->getFormDisplay('profile', $this->profileType->id(), 'default')
      ->setComponent('test_field', ['type' => 'string_textfield'])
      ->save();

    // Create a product and variation to put into the cart.
    $this->variation = $this->createEntity('commerce_product_variation', [
      'type' => 'default',
      'sku' => strtolower($this->randomMachineName()),
      'price' => [
        'number' => 9.99,
        'currency_code' => 'USD',
      ],
    ]);

    /** @var \Drupal\commerce_product\Entity\ProductInterface $product */
    $this->product = $this->createEntity('commerce_product', [
      'type' => 'default',
      'title' => 'My product',
      'variations' => [$this->variation],
      'stores' => [$this->store],
    ]);
  }

  /**
   * Tests admin management of the functionality of the module.
   */
  public function testAdminFunctions(): void {
    // Confirm the checkout flow includes the profile checkout pane.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/commerce/config/checkout-flows/manage/default');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains(
      $this->profileType->label(),
      'The profile checkout pane is shown in admin configuration page'
    );
    $pane_id = "configuration[panes][profile_form:{$this->profileType->id()}][step_id]";
    $select_field = $this->assertSession()->selectExists($pane_id);
    $this->assertEquals(
      'order_information',
      $select_field->getValue(),
      'The checkout step for profile pane is not "order_information".'
    );
  }

  /**
   * Tests a user with a profile sees it in the pane.
   */
  public function testUserWithProfile(): void {
    // Log in the test user.
    $this->drupalLogin($this->customer);

    // Create a profile for the user.
    $this->drupalGet(implode('/', [
      'user',
      $this->customer->id(),
      $this->profileType->id(),
    ]));
    $field_value = $this->randomString();
    $this->submitForm([
      "test_field[0][value]" => $field_value,
    ], 'Save');
    $this->assertSession()->statusCodeEquals(200);

    // Create a cart for the customer. No need to go via the UI.
    $cart_order = $this->container->get('commerce_cart.cart_provider')->createCart('default', $this->store, $this->customer);
    $this->cartManager->addEntity($cart_order, $this->variation);

    $this->drupalGet('cart');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm([], 'Checkout');
    $this->assertSession()->statusCodeEquals(200);

    $this->assertSession()->pageTextContains('Edit profile', "The profile pane is shown in the checkout step.");
    $this->assertSession()->pageTextContains('Profile field', "The profile field is shown in the profile checkout pane.");
    $this->assertSession()->fieldValueEquals("profile_form:{$this->profileType->id()}[profile][test_field][0][value]", $field_value);

    $field_new_value = $this->randomString();
    $this->submitForm([
      'billing_information[profile][address][0][address][given_name]' => $this->randomString(),
      'billing_information[profile][address][0][address][family_name]' => $this->randomString(),
      'billing_information[profile][address][0][address][organization]' => $this->randomString(),
      'billing_information[profile][address][0][address][address_line1]' => $this->randomString(),
      'billing_information[profile][address][0][address][postal_code]' => '94043',
      'billing_information[profile][address][0][address][locality]' => 'Mountain View',
      'billing_information[profile][address][0][address][administrative_area]' => 'CA',
      "profile_form:{$this->profileType->id()}[profile][test_field][0][value]" => $field_new_value,
    ], 'Continue to review');
    $this->assertSession()->statusCodeEquals(200);

    $this->profileStorage->resetCache();

    $profile = $this->profileStorage->loadByUser($this->customer, $this->profileType->id());

    $this->assertEquals($field_new_value, $profile->test_field->value, "The profile field value was updated.");
  }

  /**
   * Tests a user with a profile and access revoked does not see the pane.
   */
  public function testUserWithoutAccessWithProfile(): void {
    // Log in the test user.
    $this->drupalLogin($this->customer);

    // Create a profile for the user.
    $this->drupalGet(implode('/', [
      'user',
      $this->customer->id(),
      $this->profileType->id(),
    ]));
    $this->assertSession()->statusCodeEquals(200);
    $field_value = $this->randomString();
    $this->submitForm([
      "test_field[0][value]" => $field_value,
    ], 'Save');
    $this->assertSession()->statusCodeEquals(200);

    // Revoke the permissions from the role.
    $roles = $this->customer->getRoles();
    user_role_revoke_permissions($roles[1], [
      "create {$this->profileType->id()} profile",
      "update own {$this->profileType->id()} profile",
    ]);

    // Create a cart for the customer. No need to go via the UI.
    $cart_order = $this->container->get('commerce_cart.cart_provider')
      ->createCart('default', $this->store, $this->customer);
    $this->cartManager->addEntity($cart_order, $this->variation);

    $this->drupalGet('cart');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm([], 'Checkout');
    $this->assertSession()->statusCodeEquals(200);

    $this->assertSession()->pageTextNotContains(
      'Edit profile',
      'The profile pane is shown in the checkout step.'
    );
    $this->assertSession()->pageTextNotContains(
      'Profile field',
      'The profile field is shown in the profile checkout pane.'
    );
  }

  /**
   * Tests a user without a profile has one created by the pane.
   */
  public function testUserWithoutProfile(): void {
    // Log in the test user.
    $this->drupalLogin($this->customer);

    $profile = $this->profileStorage->loadByUser($this->customer, $this->profileType->id());
    $this->assertEmpty($profile, "The user does not have a {$this->profileType->id()} profile.");

    // Create a cart for the customer. No need to go via the UI.
    $cart_order = $this->container->get('commerce_cart.cart_provider')->createCart('default', $this->store, $this->customer);
    $this->cartManager->addEntity($cart_order, $this->variation);

    $this->drupalGet('cart');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm([], 'Checkout');
    $this->assertSession()->statusCodeEquals(200);

    $this->assertSession()->pageTextContains('Edit profile', "The profile pane is shown in the checkout step.");
    $this->assertSession()->pageTextContains('Profile field', "The profile field is shown in the profile checkout pane.");

    $field_value = $this->randomString();
    $this->submitForm([
      'billing_information[profile][address][0][address][given_name]' => $this->randomString(),
      'billing_information[profile][address][0][address][family_name]' => $this->randomString(),
      'billing_information[profile][address][0][address][organization]' => $this->randomString(),
      'billing_information[profile][address][0][address][address_line1]' => $this->randomString(),
      'billing_information[profile][address][0][address][postal_code]' => '94043',
      'billing_information[profile][address][0][address][locality]' => 'Mountain View',
      'billing_information[profile][address][0][address][administrative_area]' => 'CA',
      "profile_form:{$this->profileType->id()}[profile][test_field][0][value]" => $field_value,
    ], 'Continue to review');
    $this->assertSession()->statusCodeEquals(200);

    $this->profileStorage->resetCache();

    $profile = $this->profileStorage->loadByUser($this->customer, $this->profileType->id());
    $this->assertNotEmpty($profile, "The user has a {$this->profileType->id()} profile.");

    $this->assertEquals($field_value, $profile->test_field->value, "The profile field value was set.");
  }

  /**
   * Tests a user without a profile has one created by the pane.
   */
  public function testUserWithoutAccessWithoutProfile(): void {
    // Log in the test user.
    $this->drupalLogin($this->customer);

    $profile = $this->profileStorage
      ->loadByUser($this->customer, $this->profileType->id());
    $this->assertEmpty(
      $profile,
      "The user does not have a {$this->profileType->id()} profile."
    );

    // Revoke the permissions from the role.
    $roles = $this->customer->getRoles();
    user_role_revoke_permissions($roles[1], [
      "create {$this->profileType->id()} profile",
      "update own {$this->profileType->id()} profile",
    ]);

    // Create a cart for the customer. No need to go via the UI.
    $cart_order = $this->container->get('commerce_cart.cart_provider')
      ->createCart('default', $this->store, $this->customer);
    $this->cartManager->addEntity($cart_order, $this->variation);

    $this->drupalGet('cart');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm([], 'Checkout');
    $this->assertSession()->statusCodeEquals(200);

    $this->assertSession()->pageTextNotContains(
      'Edit profile',
      'The profile pane is shown in the checkout step.'
    );
    $this->assertSession()->pageTextNotContains(
      'Profile field',
      'The profile field is shown in the profile checkout pane.'
    );
  }

}
