<?php

namespace Drupal\commerce_recruiting\Controller;

use Drupal\commerce_recruiting\Entity\CampaignInterface;
use Drupal\Core\Controller\ControllerBase;

/**
 * Controller for campaign related pages.
 */
class CampaignController extends ControllerBase {

  /**
   * {@inheritdoc}
   */
  public function recruitersListPage(CampaignInterface $commerce_recruitment_campaign) {
    $campaign_option_storage = $this->entityTypeManager()->getStorage('commerce_recruitment_camp_option');
    $options = $campaign_option_storage->loadByProperties(['campaign_id' => $commerce_recruitment_campaign->id()]);
    $options = array_keys($options);
    $recruitments_storage = $this->entityTypeManager()->getStorage('commerce_recruitment');
    $query = $recruitments_storage->getQuery()->accessCheck(FALSE);
    $query->condition('campaign_option', $options, 'IN');
    $recruitment_ids = $query->execute();

    $chunk_size = 200;
    $chunks = array_chunk($recruitment_ids, $chunk_size);

    $users = [];
    $products = [];
    $data = [];
    $user_storage = $this->entityTypeManager()->getStorage('user');
    $product_storage = $this->entityTypeManager()->getStorage('commerce_product');

    foreach ($chunks as $chunk) {
      $loaded_user_ids = [];
      $loaded_product_ids = [];
      $chunk_data = [];

      /** @var \Drupal\commerce_recruiting\Entity\RecruitmentInterface[] $recruitments */
      $recruitments = $recruitments_storage->loadMultiple($chunk);

      // Collect user and product IDs first.
      foreach ($recruitments as $recruitment) {
        $user_id = $recruitment->getOwnerId();
        $loaded_user_ids[$user_id] = $user_id;

        $product = $recruitment->getProduct();
        if ($product) {
          $product_id = $product->id();
          $loaded_product_ids[$product_id] = $product_id;
        }
      }

      // Batch load users and products.
      $loaded_users = $user_storage->loadMultiple($loaded_user_ids);
      $loaded_products = $product_storage->loadMultiple($loaded_product_ids);

      // Process recruitments in this chunk.
      foreach ($recruitments as $recruitment) {
        $user_id = $recruitment->getOwnerId();
        if (!isset($users[$user_id]) && isset($loaded_users[$user_id])) {
          $users[$user_id] = $loaded_users[$user_id]->getEmail();
        }

        $product = $recruitment->getProduct();
        if ($product && isset($loaded_products[$product->id()])) {
          $product_id = $product->bundle() . '_' . $product->id();
          if (!isset($products[$product_id])) {
            $products[$product_id] = $loaded_products[$product->id()]->label();
          }

          if (!isset($chunk_data[$user_id][$product_id])) {
            $chunk_data[$user_id][$product_id] = 0;
          }
          $chunk_data[$user_id][$product_id]++;
        }
      }

      // Merge chunk data into main data array.
      foreach ($chunk_data as $user_id => $user_product_data) {
        if (!isset($data[$user_id])) {
          $data[$user_id] = [];
        }
        foreach ($user_product_data as $product_id => $count) {
          if (!isset($data[$user_id][$product_id])) {
            $data[$user_id][$product_id] = 0;
          }
          $data[$user_id][$product_id] += $count;
        }
      }

      // Clear entity cache for this chunk to free memory.
      $recruitments_storage->resetCache($chunk);
      $user_storage->resetCache($loaded_user_ids);
      $product_storage->resetCache($loaded_product_ids);
    }

    $header = [$this->t('Account')];
    foreach ($products as $product) {
      $header[] = $product;
    }

    $rows = [];
    foreach ($data as $user_id => $user_data) {
      $row = [$users[$user_id] ?? $this->t('Unknown user (@id)', ['@id' => $user_id])];
      foreach ($products as $product_id => $product) {
        $row[] = $user_data[$product_id] ?? 0;
      }

      $rows[] = $row;
    }

    usort($rows, function ($x, $y) {
      $count_x = 0;
      $count_y = 0;
      foreach ($x as $key => $col) {
        if ($key === 0) {
          continue;
        }
        $count_x += $col;
      }
      foreach ($y as $key => $col) {
        if ($key === 0) {
          continue;
        }
        $count_y += $col;
      }

      if ($count_x === $count_y) {
        return 0;
      }
      return $count_x < $count_y ? 1 : -1;
    });

    $build['table'] = [
      '#theme' => 'table',
      '#header' => $header,
      '#rows' => $rows,
    ];

    return $build;
  }

}
