<?php

namespace Drupal\Tests\commerce_recruiting\Kernel;

use Drupal\commerce_recruiting\Code;
use Drupal\Tests\commerce_recruiting\Traits\RecruitmentEntityCreationTrait;

/**
 * CampaignManager.
 *
 * @group commerce_recruiting
 */
class CampaignManagerTest extends CommerceRecruitingKernelTestBase {

  use RecruitmentEntityCreationTrait;

  /**
   * Test testFindNoRecruiterCampaigns.
   */
  public function testFindNoRecruiterCampaigns() {
    $unspecific_config = $this->createCampaign();

    $expected_product = $this->createProduct();
    $product_specific_config = $this->createCampaign(NULL, $expected_product);

    // No filter test.
    $configs = $this->campaignManager->findNoRecruiterCampaigns();
    $this->assertEquals(count($configs), 2, 'Expect two campaigns without recruiter filter.');
    $this->assertEquals($unspecific_config->id(), $configs[$unspecific_config->id()]->id(), 'Unspecific campaign is returned.');
    $this->assertEquals($product_specific_config->id(), $configs[$product_specific_config->id()]->id(), 'Product-specific campaign is returned.');

    // Product filter test.
    $configs = $this->campaignManager->findNoRecruiterCampaigns($expected_product);
    $this->assertEquals(count($configs), 1, 'Product-filtered campaign count matches.');
    $this->assertEquals($product_specific_config->id(), $configs[$product_specific_config->id()]->id(), 'Product-specific campaign returned for product filter.');
  }

  /**
   * Test testFindRecruiterCampaigns.
   */
  public function testFindRecruiterCampaigns() {
    $user = $this->drupalCreateUser();
    $recruiter_config = $this->createCampaign($user);

    $differnt_product = $this->createProduct();
    $recruiter_product_config = $this->createCampaign($user, $differnt_product);

    // User filter test.
    $configs = $this->campaignManager->findRecruiterCampaigns($user);
    $this->assertEquals(count($configs), 2, 'Expect recruiter campaigns for user.');
    $this->assertEquals($recruiter_config->id(), $configs[$recruiter_config->id()]->id(), 'Recruiter campaign returned.');
    $this->assertEquals($recruiter_product_config->id(), $configs[$recruiter_product_config->id()]->id(), 'Recruiter product-specific campaign returned.');
  }

  /**
   * Test testGetSessionFromCode.
   */
  public function testGetSessionFromCode() {
    $friend_campaign = $this->createCampaign();
    $friend_recruiter = $this->createUser();
    $code_string = $friend_campaign->getFirstOption()->getCode();
    $code = Code::createFromCode($code_string . '--' . $friend_recruiter->id());
    $session = $this->campaignManager->saveRecruitmentSession($code);
    $this->assertEquals($session->getRecruiter()->id(), $friend_recruiter->id(), 'Friend recruiter stored in session.');
    $this->assertEquals($session->getCampaignOption()->id(), $friend_campaign->getFirstOption()->id(), 'Friend campaign option stored in session.');

    $recruiter_recruiter = $this->createUser();
    $recruiter_campaign = $this->createCampaign($recruiter_recruiter);
    $code_string = $recruiter_campaign->getFirstOption()->getCode();
    $code = Code::createFromCode($code_string);
    $recruiter_session = $this->campaignManager->saveRecruitmentSession($code);
    $this->assertEquals($recruiter_session->getRecruiter()->id(), $recruiter_campaign->getRecruiter()->id(), 'Recruiter session uses recruiter campaign recruiter.');
    $this->assertEquals($recruiter_session->getCampaignOption()->id(), $recruiter_campaign->getFirstOption()->id(), 'Recruiter session uses recruiter campaign option.');

  }

  /**
   * Test getRecruitmentInfoFromCode.
   */
  public function testGetRecruitmentInfoFromCode() {
    $campaign = $this->createCampaign($this->drupalCreateUser());
    $code_string = $campaign->getFirstOption()->getCode();
    $code = Code::createFromCode($code_string);
    $this->assertNotNull($code, 'Code created from campaign option string.');
    $option = $this->campaignManager->findCampaignOptionFromCode($code);
    $this->assertNotNull($option, 'Option resolved from code.');
    $this->assertNotNull($option->getCampaign()->getRecruiter(), 'Option recruiter present.');
    $this->assertEquals($campaign->getFirstOption()->id(), $option->id(), 'Resolved option matches campaign first option.');
  }

}
