<?php

namespace Drupal\Tests\commerce_recruiting\Kernel;

use Drupal\commerce_price\Price;
use Drupal\commerce_recruiting\Entity\CampaignOptionInterface;
use Drupal\commerce_recruiting\RecruitmentSession;
use Drupal\Tests\commerce_recruiting\Traits\RecruitmentEntityCreationTrait;

// @codingStandardsIgnoreStart
/**
 * RecruitmentManagerTest.
 *
 * @group commerce_recruiting
 */
class RecruitmentManagerTest extends CommerceRecruitingKernelTestBase {

  use RecruitmentEntityCreationTrait;

  /**
   * Test sessionMatch.
   */
  public function testSessionMatch() {
    $recruiter = $this->createUser();
    $product_a = $this->createProduct();
    $product_b = $this->createProduct();
    $order_a = $this->createOrder([$product_a]);
    $order_b = $this->createOrder([$product_b]);
    $order_c = $this->createOrder([$product_a, $product_b]);

    // Test campaign with 'bonus for any option' off.
    $campaign = $this->createCampaign($recruiter, NULL, FALSE, FALSE);
    $option_a = $this->createCampaignOption($product_a);
    $option_b = $this->createCampaignOption($product_b);
    $campaign->addOption($option_a)->addOption($option_b);
    $campaign->save();

    $prophecy = $this->prophesize(RecruitmentSession::CLASS);
    $prophecy->getCampaignOption()->willReturn($campaign->getFirstOption());
    $prophecy->getRecruiter()->willReturn($recruiter);
    \Drupal::getContainer()->set('commerce_recruiting.recruitment_session', $prophecy->reveal());

    $this->recruitmentManager = $this->container->get('commerce_recruiting.recruitment_manager');
    $this->assertCount(1, $this->recruitmentManager->sessionMatch($order_a), 'One match when only product A is present.');
    $this->assertCount(0, $this->recruitmentManager->sessionMatch($order_b), 'No match when only product B with bonus_any_option FALSE.');
    $this->assertCount(1, $this->recruitmentManager->sessionMatch($order_c), 'One match when order contains product A.');

    // Test with 'bonus for any option' on.
    $campaign = $this->createCampaign($recruiter, NULL, FALSE);
    $option_a = $this->createCampaignOption($product_a);
    $option_b = $this->createCampaignOption($product_b);
    $campaign->addOption($option_a)->addOption($option_b);
    $campaign->save();

    $prophecy = $this->prophesize(RecruitmentSession::CLASS);
    $prophecy->getCampaignOption()->willReturn($campaign->getFirstOption());
    $prophecy->getRecruiter()->willReturn($recruiter);
    \Drupal::getContainer()->set('commerce_recruiting.recruitment_session', $prophecy->reveal());

    $this->assertCount(1, $this->recruitmentManager->sessionMatch($order_a), 'Match order A when bonus_any_option TRUE.');
    $this->assertCount(1, $this->recruitmentManager->sessionMatch($order_b), 'Match order B when bonus_any_option TRUE.');
    $this->assertCount(2, $this->recruitmentManager->sessionMatch($order_c), 'Both products match when bonus_any_option TRUE.');
  }

  /**
   * Test applyTransitions.
   */
  public function testApplyTransitions() {
    $recruiter = $this->createUser();
    $campaign = $this->createCampaign($recruiter);
    $recruited = $this->createUser();
    $products = [];
    $products[] = $this->createProduct();
    $products[] = $this->createProduct();
    $order = $this->createOrder($products, 'draft');

    $recruitments = $this->createRecruitings($campaign, $recruiter, $recruited, $order);

    $this->assertCount(2, $recruitments, 'Two recruitments created for draft order.');
    $this->recruitmentManager->applyTransitions("accept");
    $items = $this->entityTypeManager->getStorage('commerce_recruitment')->loadByProperties(['state' => 'accepted']);
    $this->assertCount(0, $items, 'No items in accepted state when applying accept transition without state changes.');
  }

  /**
   * Test resolveRecruitmentBonus.
   */
  public function testResolveRecruitmentBonus() {
    // Campaign test with quantity multiplicator.
    $campaign = $this->createCampaign(NULL, NULL, FALSE);

    // First product and option to test fix bonus calculation.
    $product_a = $this->createProduct();
    $option_a = $this->createCampaignOption($product_a, '15');
    $campaign->addOption($option_a);
    $order_item_a = $this->createOrderItem($product_a, 2);

    // Second product with second option to test percent calculation.
    $product_b = $this->createProduct(new Price(15, 'USD'));
    $option_b = $this->createCampaignOption($product_b, '10', CampaignOptionInterface::RECRUIT_BONUS_METHOD_PERCENT);
    $campaign->addOption($option_b);
    $order_item_b = $this->createOrderItem($product_b, 2);
    $campaign->save();

    $options = $campaign->getOptions();
    $this->assertCount(2, $options, 'Two options attached to campaign.');
    $bonus_a = $this->recruitmentManager->resolveRecruitmentBonus($options[0], $order_item_a);
    // 2 * 15 = 30.
    $this->assertEquals(30, $bonus_a->getNumber(), 'Fixed bonus multiplied by quantity equals 30.');

    $bonus_b = $this->recruitmentManager->resolveRecruitmentBonus($options[1], $order_item_b);
    // 2 * 15 / 10 = 3.
    $this->assertEquals(3, $bonus_b->getNumber(), 'Percent bonus with multiplication equals 3.');


    // Campaign test without quantity multiplicator.
    $campaign = $this->createCampaign(NULL, NULL, FALSE, FALSE, FALSE);

    // First product and option to test fix bonus calculation.
    $product_a = $this->createProduct();
    $option_a = $this->createCampaignOption($product_a, '10');
    $campaign->addOption($option_a);
    $order_item_a = $this->createOrderItem($product_a, 2);

    // Second product with second option to test percent calculation.
    $product_b = $this->createProduct(new Price(10, 'USD'));
    $option_b = $this->createCampaignOption($product_b, '50', CampaignOptionInterface::RECRUIT_BONUS_METHOD_PERCENT);
    $campaign->addOption($option_b);
    $order_item_b = $this->createOrderItem($product_b, 2);
    $campaign->save();

    $options = $campaign->getOptions();
    $this->assertCount(2, $options, 'Two options on campaign without quantity multiplication.');
    $bonus_a = $this->recruitmentManager->resolveRecruitmentBonus($options[0], $order_item_a);
    // Fix bonus 10.
    $this->assertEquals(10, $bonus_a->getNumber(), 'Fixed bonus without multiplication remains 10.');

    $bonus_b = $this->recruitmentManager->resolveRecruitmentBonus($options[1], $order_item_b);
    // 50% of 10.
    $this->assertEquals(5, $bonus_b->getNumber(), 'Percent bonus without multiplication equals 5.');
  }

  /**
   * Test getRecruitmentsByOrder.
   */
  public function testGetRecruitmentsByOrder() {
    $recruiter = $this->createUser();
    $recruited = $this->createUser();
    $product1 = $this->createProduct();
    $product2 = $this->createProduct();
    $campaign = $this->createCampaign($recruiter, $product1);
    $option2 = $this->createCampaignOption($product2);
    $campaign->addOption($option2);
    $campaign->save();

    $order = $this->createOrder([$product1]);
    $recruitments = $this->createRecruitings($campaign, $recruiter, $recruited, $order);
    $this->assertCount(1, $recruitments, 'One recruitment created for single product order.');
    $found_recruitment = $this->recruitmentManager->getRecruitmentsByOrder($order);
    $this->assertCount(1, $found_recruitment, 'Count of loaded recruitments by order.');
    $this->assertEquals(current($recruitments)->uuid(), current($found_recruitment)->uuid(), 'Recruitments match');
  }

  /**
   * Test getRecruitmentSummaryByCampaign.
   */
  public function testGetRecruitmentSummaryByCampaign() {
    $recruiter = $this->createUser();
    $product1 = $this->createProduct();
    $product2 = $this->createProduct();
    $product3 = $this->createProduct();
    $campaign = $this->createCampaign($recruiter, $product1);
    $recruited = $this->createUser();
    $campaign2 = $this->createCampaign($recruiter);
    $productc21 = $this->createProduct();
    $productc22 = $this->createProduct();
    $productc23 = $this->createProduct();

    $this->createRecruitings($campaign, $recruiter, $recruited, $this->createOrder([$product1, $product2, $product3]));
    $recruitments2 = $this->createRecruitings($campaign2, $recruiter, $recruited, $this->createOrder([$productc21, $productc22, $productc23]));
    $this->assertCount(3, $recruitments2, 'Three recruitments created for second campaign.');
    $this->recruitmentManager->applyTransitions("accept");
    $summary = $this->recruitmentManager->getRecruitmentSummaryByCampaign($campaign,'accepted', $recruiter);
    $this->assertCount(3, $summary->getResults(), 'Summary returns three recruitments for campaign.');
    $this->assertEquals(30, $summary->getTotalPrice()->getNumber(), 'Summary total price equals 30.');
  }

  /**
   * Test findRecruitmentsByCampaign.
   */
  public function testFindRecruitmentsByCampaign() {
    $recruiter = $this->createUser();
    $product1 = $this->createProduct();
    $product2 = $this->createProduct();
    $product3 = $this->createProduct();
    $campaign = $this->createCampaign($recruiter, $product1);
    $recruited = $this->createUser();
    $campaign2 = $this->createCampaign($recruiter);
    $productc21 = $this->createProduct();
    $productc22 = $this->createProduct();
    $productc23 = $this->createProduct();

    $recruitments = $this->createRecruitings($campaign, $recruiter, $recruited, $this->createOrder([$product1, $product2, $product3]));
    $recruitments2 = $this->createRecruitings($campaign2, $recruiter, $recruited, $this->createOrder([$productc21, $productc22, $productc23]));
    $this->assertCount(3, $recruitments, 'Three recruitments exist for first campaign.');
    $found_recruitments = $this->recruitmentManager->findRecruitmentsByCampaign($campaign, 'created', $recruiter);
    $this->assertCount(3, $found_recruitments, 'Find recruitments returns three results for created state.');
  }

}
