<?php

namespace Drupal\commerce_satispay\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\Core\Form\FormStateInterface;
use SatispayGBusiness\Payment;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\DependencyInjection\ContainerInterface;
use SatispayGBusiness\Api;

/**
 * Provides the Satispay payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "satispay_online",
 *   label =  @Translation("Satispay"),
 *   display_label = @Translation("Satispay"),
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_satispay\PluginForm\SatispayOnlineForm",
 *   },
 *   payment_method_types = {"credit_card"},
 *   credit_card_types = {
 *     "amex", "dinersclub", "discover", "jcb", "maestro", "mastercard", "visa",
 *   },
 * )
 */
class SatispayOnline extends OffsitePaymentGatewayBase implements SatispayOnlineInterface {

  /**
   * The logger channel for this module..
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new PaymentGatewayBase object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\commerce_payment\PaymentTypeManager $payment_type_manager
   *   The payment type manager.
   * @param \Drupal\commerce_payment\PaymentMethodTypeManager $payment_method_type_manager
   *   The payment method type manager.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_channel_factory
   *   The logger channel factory.
   */

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->logger = $container->get('logger.factory')->get('commerce_satispay');
    $instance->messenger =  $container->get('messenger');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
        'satispay_public_key' => '',
        'satispay_private_key' => '',
        'satispay_key_id' => '',

      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['satispay_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Satispay Token'),
      '#description' => $this->t('Use this field to initialize the system the first time.<br/>
                   Activation code that can be generated from the Satispay Dashboard
                   (or provided manually for Sandbox account)<br/>
                   Remember that the token is disposable.</br>
                   <strong>By entering a valid code, Drupal will automatically generate all the necessary keys and save them in the fields below.</strong>
                   '),
      '#default_value' => '',
      '#required' => FALSE,
      '#maxlength' => 10,
      '#size' => 10,
    ];


    $form['satispay_public_key'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Satispay Public Key'),
      '#default_value' => $this->configuration['satispay_public_key'],
      '#required' => FALSE,
      '#maxlength' => 1000,
      '#size' => 150,
      '#description' => $this->t('A valid key startd with "-----BEGIN PUBLIC KEY-----" and endd with "-----END PUBLIC KEY-----"'),
    ];
    $form['satispay_private_key'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Satispay Private Key'),
      '#default_value' => $this->configuration['satispay_private_key'],
      '#required' => FALSE,
      '#maxlength' => 4000,
      '#size' => 150,
      '#description' => $this->t('A valid key startd with "-----BEGIN PRIVATE KEY-----" and endd with "-----END PRIVATE KEY-----"'),
    ];
    $form['satispay_key_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Satispay Key ID'),
      '#default_value' => $this->configuration['satispay_key_id'],
      '#required' => FALSE,
      '#maxlength' => 500,
      '#size' => 150,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);
    if (!$form_state->getErrors() && $form_state->isSubmitted()) {
      $values = $form_state->getValue($form['#parents']);
      if($values['satispay_token']){
        //First Time: All field must be empty
        if(strlen($values['satispay_public_key']) > 0 ||
           strlen($values['satispay_private_key']) > 0 ||
           strlen($values['satispay_key_id']) > 0){
          $form_state->setError($form['satispay_public_key'], $this->t('To generate new keys these fields must be empty.'));
          $form_state->setError($form['satispay_private_key'], $this->t('To generate new keys these fields must be empty.'));
          $form_state->setError($form['satispay_key_id'], $this->t('To generate new keys these fields must be empty.'));
        }
      }
      else{
        if(strlen($values['satispay_public_key']) == 0){
          $form_state->setError($form['satispay_public_key'], $this->t('Field must be completed'));
        }
        if(strlen($values['satispay_private_key']) == 0){
          $form_state->setError($form['satispay_private_key'], $this->t('Field must be completed'));
        }
        if(strlen($values['satispay_key_id']) == 0){
          $form_state->setError($form['satispay_key_id'], $this->t('Field must be completed'));
        }

      }

      /*
      if (!$this->check()) {
        $form_state->setError($form['satispay_public_key'], $this->t('Invalid satispay_public_key'));
        $form_state->setError($form['satispay_private_key'], $this->t('Invalid satispay_private_key'));
        $form_state->setError($form['satispay_key_id'], $this->t('Invalid satispay_key_id'));
      }*/
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    $values = $form_state->getValue($form['#parents']);
    if(strlen($values['satispay_token']) == 0){
      $this->configuration['satispay_public_key'] = $values['satispay_public_key'];
      $this->configuration['satispay_private_key'] = $values['satispay_private_key'];
      $this->configuration['satispay_key_id'] = $values['satispay_key_id'];
      $this->messenger->addMessage($this->t('The Satispay keys have been saved.'));
    }
    else{
      if ($values['mode'] == 'test') {
        \SatispayGBusiness\Api::setSandbox(TRUE);
      }
      try{
        $authentication = \SatispayGBusiness\Api::authenticateWithToken($values['satispay_token']);
        $this->configuration['satispay_public_key'] = $authentication->publicKey;
        $this->configuration['satispay_private_key'] = $authentication->privateKey;
        $this->configuration['satispay_key_id'] = $authentication->keyId;
        $this->messenger->addMessage($this->t('The new Satispay keys have been generated and saved.'));
      }
      catch (\Exception $e) {
        $this->messenger->addError($this->t('The new keys could not be generated. Please check the token and try again.'));
        $this->logger->error('Satispay token authentication failed: @message', ['@message' => $e->getMessage()]);
      }
    }

  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    $data = $order->getData('satispay');
    $id = $data['uuid'];
    $this->logger->debug('Satispay onReturn uuid: @uuid',['@uuid' => $id] );
    $this->configureAPI();
    try {
      $satispay_payment = Payment::get($id);
    } catch (\Exception $e) {
      $this->logger->error('Satispay error for @id: @msg', [
        '@id' => $id,
        '@msg' => $e->getMessage(),
      ]);
      throw new PaymentGatewayException("Satispay Payment error on get data");
    }
    $payment = $this->entityTypeManager->getStorage('commerce_payment')
      ->loadByRemoteId($satispay_payment->id);
    // Update the payment state.
    switch ($satispay_payment->status) {
      case 'PENDING':
        $payment->setState('pending');
        break;

      case 'ACCEPTED':
        $payment->setState('completed');
        break;
      case 'CANCELED':
        $payment->setState('failed');
        break;
    }
    $payment->setRemoteState($satispay_payment->status);
    $payment->save();
    $this->logger->debug('Satispay onReturn completed: @msg',['@msg' => json_encode($satispay_payment)]);
    if ($satispay_payment->status != 'ACCEPTED') {
      $this->logger->error('Satispay transaction @id not accepted (aborted by user)', ['@id' => $id,]);
      throw new PaymentGatewayException("Satispay payment was not accepted");
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request) {
    $uuid = $request->get('uuid');
    $this->logger->debug('Satispay onNotify uuid: @uuid',['@uuid' => $uuid] );
    if(is_null($uuid)){
      $this->logger->alert('The onNotify request does not have a uuid. Ignored.');
      return FALSE;
    }
    $payment = $this->entityTypeManager->getStorage('commerce_payment')
      ->loadByRemoteId($uuid);

    if(is_null($payment)){
      //The payment was initialized at the beginning by the redirect form, there must be a UUID
      $this->logger->alert('The onNotify request does not have a  valid payment. Ignored.');
      return FALSE;
    }

    $this->configureAPI();
    try {
      $satispay_payment = Payment::get($uuid);
    } catch (\Exception $e) {
      $this->logger->error('Satispay error for @id: @msg', [
        '@id' => $uuid,
        '@msg' => $e->getMessage(),
      ]);
      return FALSE;
    }
    switch ($satispay_payment->status) {
      case 'PENDING':
        $payment->setState('pending');
        break;

      case 'ACCEPTED':
        $payment->setState('completed');
        break;
      case 'CANCELED':
        $payment->setState('failed');
        break;
    }
    $payment->setRemoteState($satispay_payment->status);
    //$payment->setAmount($satispay_payment->amount_unit);
    $payment->save();
    $this->logger->debug('Satispay onNotify completed: @msg',['@msg' => json_encode($satispay_payment)]);

  }

  /**
   * {@inheritdoc}
   */
  public function check() {
    $this->configureAPI();
    try {
      $payments = Payment::all();
      return TRUE;
    } catch (\Exception $ex) {
      return FALSE;
    }
  }


  /*
   * Simple function to setup the Satispay Connection
   */
  public function configureAPI() {
    $configuration = $this->getConfiguration();
    if ($configuration['mode'] == 'test') {
      Api::setSandbox(TRUE);
    }
    Api::setPlatformHeader('Drupal');
    Api::setPlatformVersionHeader(\Drupal::VERSION);
    Api::setPluginNameHeader('Commerce Satispay');
    Api::setPluginVersionHeader('2.x');
    Api::setPublicKey($configuration['satispay_public_key']);
    Api::setPrivateKey($configuration['satispay_private_key']);
    Api::setKeyId($configuration['satispay_key_id']);
  }


  /**
   * {@inheritdoc}
   */
  public function setCheckout(PaymentInterface $payment, array $extra) {
    $order = $payment->getOrder();
    $amount = $payment->getAmount()->convert('EUR');
    $user = \Drupal::currentUser();
    $this->configureAPI();

    $notify_url = $this->getNotifyUrl();
    $notify_url->setOption('query', ['uuid' => '{uuid}']);

    $callback_url = $notify_url->toString();
    $callback_url = str_replace('uuid=%7Buuid%7D', 'uuid={uuid}', $callback_url);

    $payment_data = [
      "flow" => "MATCH_CODE",
      "amount_unit" => $amount->multiply(100)->getNumber(),
      //Satispay use only EUR if you use other the response is "codice 36"
      "currency" => 'EUR',
      "external_code" => $order->id(),
      'callback_url' => $callback_url,
      'redirect_url' => $extra['return_url'],
      'metadata' => [
        "order_id" => $order->id(),
        "user" => $user->getDisplayName(),
      ],
    ];

    try {
      $payment = Payment::create($payment_data);
    } catch (\Exception $e) {
      $this->logger->error('Satispay error:  @msg', ['@msg' => $e->getMessage()]);
      throw new PaymentGatewayException("Satispay Payment Error: API throw Exception");
    }

    return $payment;
  }

}
