<?php

namespace Drupal\commerce_shipping;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the list builder for shipments.
 */
class ShipmentListBuilder extends EntityListBuilder {

  /**
   * The currency formatter.
   *
   * @var \CommerceGuys\Intl\Formatter\CurrencyFormatterInterface
   */
  protected $currencyFormatter;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * {@inheritdoc}
   */
  protected $entitiesKey = 'shipments';

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    $instance = parent::createInstance($container, $entity_type);
    $instance->currencyFormatter = $container->get('commerce_price.currency_formatter');
    $instance->routeMatch = $container->get('current_route_match');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'commerce_shipments';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEntityIds() {
    $order_id = $this->routeMatch->getRawParameter('commerce_order');
    $query = $this->getStorage()->getQuery()
      ->accessCheck(FALSE)
      ->condition('order_id', $order_id)
      ->sort($this->entityType->getKey('id'));

    // Only add the pager if a limit is specified.
    if ($this->limit) {
      $query->pager($this->limit);
    }
    return $query->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header = [
      'label' => $this->t('Shipment'),
      'tracking' => $this->t('Tracking'),
      'amount' => $this->t('Amount'),
      'state' => $this->t('State'),
    ];
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface $entity */
    $amount = $entity->getAmount();

    $row['label']['data'] = [
      '#type' => 'link',
      '#title' => $entity->label(),
      '#url' => $entity->toUrl(),
      '#options' => $entity->toUrl()->getOptions(),
    ];
    $row['tracking'] = $entity->getTrackingCode();
    $row['amount'] = !empty($amount) ? $this->currencyFormatter->format($amount->getNumber(), $amount->getCurrencyCode()) : '';
    $row['state'] = $entity->getState()->getLabel();

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  protected function getDefaultOperations(EntityInterface $entity) {
    $operations = parent::getDefaultOperations($entity);
    $order = $entity->getOrder();

    if ($order && $order->access('resend_receipt')) {
      $operations['resend_confirmation'] = [
        'title' => $this->t('Resend confirmation'),
        'weight' => 20,
        'url' => $entity->toUrl('resend-confirmation-form'),
      ];
    }

    return $operations;
  }

}
