<?php

namespace Drupal\commerce_shipping_label_zebra\EventSubscriber;

use Drupal\commerce_shipping_label\Event\ShippingLabelEvents;
use Drupal\commerce_shipping_label\Event\ShippingLabelListEvent;
use Drupal\commerce_shipping_label\ShippingLabelManager;
use Drupal\commerce_shipping_label_zebra\Form\PrintForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Render\RendererInterface;


class ShippingLabelListSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * @param \Drupal\Core\Form\FormBuilderInterface $formBuilder
   */
  public function __construct(\Drupal\Core\Form\FormBuilderInterface $formBuilder, FileUrlGeneratorInterface $file_url_generator, RendererInterface $renderer) {
    $this->formBuilder = $formBuilder;
    $this->fileUrlGenerator = $file_url_generator;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = [
      ShippingLabelEvents::SHIPMENT_LIST_BUILDER_HEADER => ['updateHeader', 75],
      ShippingLabelEvents::SHIPMENT_LIST_BUILDER_ROW => ['updateRow', 75],
    ];

    return $events;
  }

  public function updateHeader(ShippingLabelListEvent $event) {
    $header = $event->getData();
    $header = array_slice($header, 0, count($header) - 1, TRUE) +
      ['print' => $this->t('Print')] +
      array_slice($header, count($header) - 1, 1, TRUE);
    $event->setData($header);
  }

  public function updateRow(ShippingLabelListEvent $event) {
    $shipment = $event->getShipment();
    $row = $event->getData();
    $url = NULL;
    if ($shipment->hasField('shipment_label') && !$shipment->get('shipment_label')->isEmpty()) {
      /** @var \Drupal\file\FileInterface[] $files */
      $files = $shipment->get('shipment_label')->referencedEntities();
      foreach ($files as $file) {
        if (preg_match('/\.zpl$/', $file->getFilename())) {
          $url = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
          break;
        }
      }
    }
    if ($url !== NULL) {
      $form = $this->formBuilder->getForm(PrintForm::class, $url);
      $row = array_slice($row, 0, count($row) - 1, TRUE) +
        ['print' => $this->renderer->render($form)] +
        array_slice($row, count($row) - 1, 1, TRUE);
    }
    else {
      $row = array_slice($row, 0, count($row) - 1, TRUE) +
        ['print' => ''] +
        array_slice($row, count($row) - 1, 1, TRUE);
    }
    $event->setData($row);
  }

}
